# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) 2016-2017, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# Python Imports
import json
# Local Imports
from neohost_common import NeoHostCommon
from neohost_exceptions import MCorePluginError, MMethodNotFound, MException
from neohost_plugin_ifc import AbsNeoHostCommandIFC, AbsNeoHostPluginIFC, \
    EnumCmdExecMode, EnumCmdScope, EnumCmdType
from cpp_process import MCppProcess
from plugin_manager import MPluginManager
from neohost_request import NeoHostRequest
from jsonrpc_err import JsonRpcErr
from job_mgr import JobMgr
from common_meta import MEnumMeta
from get_file_cmds import GetApplicationFileMetaData, GetApplicationFileData
from core_plugin_defs import CorePluginDefs


class SetApplicationTerminateJobCommand(AbsNeoHostCommandIFC):
    """Class that implements GetApplicationSupportedModules()
     NeoHost backend API command"""
    job_mgr = JobMgr()

    def __init__(self):
        super(SetApplicationTerminateJobCommand, self).__init__(
            cmd_name="SetApplicationTerminateJob",
            cmd_desc="Terminate job",
            cmd_type=EnumCmdType.Cmd_Type_Set,
            cmd_scope=EnumCmdScope.Cmd_Scope_Application,
            supp_exec_mask=EnumCmdExecMode.Cmd_Exec_Mode_Sync,
            cmd_in_desc="jobToken - string\nforce - bool",
            cmd_out_desc="Success",
            extra_str="")

    def execute_command(self, json_request, exec_opt):
        """
        Command execution point
        :param json_request:
        :param exec_opt:
        :return:
        """
        params = json.loads(json_request)
        job_token = params["jobToken"]
        found, error = self.job_mgr.terminateJob(job_token)
        if found:
            if not error:
                return CorePluginDefs.CORE_SUCCESS, json.dumps("Success")
            raise MCorePluginError("%s" % error)

        # send command to backend process
        request = NeoHostRequest()
        request.module = CorePluginDefs.CORE_PLUGIN_NAME
        request.method = self.get_command_name()
        request.params = json.dumps(dict(jobToken=job_token))
        result = MCppProcess().send_request(request)
        result = json.loads(result)
        rc = CorePluginDefs.CORE_SUCCESS
        if "error" in result:
            rc = CorePluginDefs.CORE_FAILURE
            result = result["error"]
        else:
            result = json.dumps(result["result"])
        return rc, result


class GetApplicationSupportedModulesCommand(AbsNeoHostCommandIFC):
    """Class that implements GetApplicationSupportedModules()
     NeoHost backend API command"""
    def __init__(self):
        super(GetApplicationSupportedModulesCommand, self).__init__(
            cmd_name="GetApplicationSupportedModules",
            cmd_desc="Get a list of available modules",
            cmd_type=EnumCmdType.Cmd_Type_Get,
            cmd_scope=EnumCmdScope.Cmd_Scope_Application,
            supp_exec_mask=EnumCmdExecMode.Cmd_Exec_Mode_All,
            cmd_in_desc="",
            cmd_out_desc="[<moduleStr1>, <moduleStr2>, ...]",
            extra_str="")

    def execute_command(self, json_request, exec_opt):
        """
        Command execution point
        :param json_request:
        :param exec_opt:
        :return:
        """
        # send command to backend process
        request = NeoHostRequest()
        request.module = CorePluginDefs.CORE_PLUGIN_NAME
        request.method = self.get_command_name()
        request.params = json.dumps(None)

        result = MCppProcess().send_request(request)
        result = json.loads(result)
        if "error" in result:
            return json.dumps(result)  # operation failed
        result = result["result"]
        python_plugin_list = MPluginManager().get_plugin_list()
        # remove unsupported python plugins
        supp_python_plugin_list = []
        for module_name in python_plugin_list:
            plugin_obj = MPluginManager().get_plugin(module_name)
            p_ver = plugin_obj.get_plugin_version()
            try:
                NeoHostCommon.check_python_plugin_version(p_ver[0], p_ver[1])
                supp_python_plugin_list.append(module_name)
            except MException:
                pass
        # append modules lists and remove duplications
        result.extend(supp_python_plugin_list)
        result = list(set(result))
        return CorePluginDefs.CORE_SUCCESS, json.dumps(result)


class GetApplicationModuleMetaDataCommand(AbsNeoHostCommandIFC):
    """Class that implements GetApplicationModuleMetaData() NeoHost
    backend API command"""
    def __init__(self):
        super(GetApplicationModuleMetaDataCommand, self).__init__(
            cmd_name="GetApplicationModuleMetaData",
            cmd_desc="Get module meta data",
            cmd_type=EnumCmdType.Cmd_Type_Get,
            cmd_scope=EnumCmdScope.Cmd_Scope_Application,
            supp_exec_mask=EnumCmdExecMode.Cmd_Exec_Mode_All,
            cmd_in_desc="<module name>",
            cmd_out_desc="object describing the module",
            extra_str="")

    def get_python_module_metadata(self, module_name):
        """Returns metadata of a python plugin"""
        result = {}
        plugin_obj = MPluginManager().get_plugin(module_name)
        # Check plugin version
        p_ver = plugin_obj.get_plugin_version()
        NeoHostCommon.check_python_plugin_version(p_ver[0], p_ver[1])

        result["name"] = plugin_obj.get_plugin_name()
        result["description"] = plugin_obj.get_plugin_description()
        p_ver = plugin_obj.get_plugin_version()
        result["version"] = "%02d.%02d" % (p_ver[0], p_ver[1])
        result["methods"] = plugin_obj.get_command_list()
        return CorePluginDefs.CORE_SUCCESS, result

    def get_module_metadata(self, module_name, get_py):
        """Returns metadata of the plugin: could be implemented in both
        cpp and python"""
        # get cpp core module meta data
        rc, module_metadata = self.get_cpp_module_metadata(module_name)
        if get_py:
            py_rc, python_module_metadata = self.get_python_module_metadata(
                module_name)
            if py_rc == CorePluginDefs.CORE_SUCCESS:
                if rc == CorePluginDefs.CORE_SUCCESS:
                    module_metadata["methods"].extend(
                        python_module_metadata["methods"])
                    module_metadata["methods"] = list(
                        set(module_metadata["methods"]))
                else:
                    module_metadata = python_module_metadata
                    rc = py_rc
        return rc, json.dumps(module_metadata)

    def get_cpp_module_metadata(self, module_name):
        """Returns metadata of a C++ plugin by calling the backend process"""
        # send command to backend process
        request = NeoHostRequest()
        request.module = CorePluginDefs.CORE_PLUGIN_NAME
        request.method = self.get_command_name()
        request.params = json.dumps(dict(module=module_name))
        result = MCppProcess().send_request(request)
        result = json.loads(result)
        rc = CorePluginDefs.CORE_SUCCESS
        if "error" in result:
            rc = CorePluginDefs.CORE_FAILURE
            result = result["error"]
        else:
            result = result["result"]
        return rc, result

    def execute_command(self, json_request, exec_opt):
        """Command execution point"""
        request = json.loads(json_request)
        module_name = request["module"]
        # check if its python module
        get_py_data = False
        if module_name in MPluginManager().get_plugin_list():
            get_py_data = True
        return self.get_module_metadata(module_name, get_py_data)


class GetApplicationMethodMetaDataCommand(AbsNeoHostCommandIFC):
    """Class that implements GetApplicationMethodMetaData() NeoHost
    backend API command."""
    class EnumFormatType(object):
        """Command type enum."""
        __metaclass__ = MEnumMeta
        Format_HumanReadable = 0
        Format_Json = 1
        Format_Xml = 2
        FormatMapping = {
            Format_HumanReadable: "Human Readable",
            Format_Json: "JSON",
            Format_Xml: "XML",
        }

    @classmethod
    def _validateFormat(cls, fmt):
        if fmt not in cls.EnumFormatType.FormatMapping:
            raise MCorePluginError(
                "Unknown input/output metadata format: (%d)" % fmt)
        if fmt != cls.EnumFormatType.Format_HumanReadable:
            fmt_str = cls.EnumFormatType.FormatMapping[fmt]
            raise MCorePluginError(
                "Unsupported input/output metadata format: %s (%d)" %
                (fmt_str, fmt))

    def __init__(self):
        super(GetApplicationMethodMetaDataCommand, self).__init__(
            cmd_name="GetApplicationMethodMetaData",
            cmd_desc="get method meta data",
            cmd_type=EnumCmdType.Cmd_Type_Get,
            cmd_scope=EnumCmdScope.Cmd_Scope_Application,
            supp_exec_mask=EnumCmdExecMode.Cmd_Exec_Mode_All,
            cmd_in_desc="<moduleName> <methodName> [format]",
            cmd_out_desc="object describing the method",
            extra_str="")

    def get_python_command_metadata(self, module_name, method_name, fmt):
        """Returns metadata of a specific python plugin command."""
        plugin_obj = MPluginManager().get_plugin(module_name)
        # Check plugin version
        p_ver = plugin_obj.get_plugin_version()
        NeoHostCommon.check_python_plugin_version(p_ver[0], p_ver[1])

        command_obj = plugin_obj.get_command(method_name)
        result = dict()
        result["name"] = command_obj.get_command_name()
        result["description"] = command_obj.get_command_desc()
        result["input"] = command_obj.get_command_input_desc()
        result["output"] = command_obj.get_command_output_desc()
        result["type"] = command_obj.get_command_type()
        result["scope"] = command_obj.get_command_scope()
        result["suppExecMode"] = command_obj.get_command_supp_exec_modes()
        result["extra"] = command_obj.get_command_extra_str()
        cli_meta_data = command_obj.get_command_cli_metadata()
        if cli_meta_data is not None:
            result["cliMetaData"] = cli_meta_data
        gui_meta_data = command_obj.get_command_gui_metadata()
        if gui_meta_data is not None:
            result["guiMetaData"] = gui_meta_data
        return CorePluginDefs.CORE_SUCCESS, json.dumps(result)

    def get_cpp_command_metadata(self, module_name, method_name, fmt,
                                 py_plugin_found):
        """Returns metadata of a specific C++ plugin command."""
        request = NeoHostRequest()
        request.module = CorePluginDefs.CORE_PLUGIN_NAME
        request.method = self.get_command_name()
        params_dict = dict(module=module_name, method=method_name)
        if fmt is not None:
            params_dict["format"] = fmt
        request.params = json.dumps(params_dict)
        result = MCppProcess().send_request(request)
        result = json.loads(result)
        err_result = result.get("error")
        if err_result:
            if py_plugin_found:
                code = err_result["code"]
                if code == JsonRpcErr.MethodNotFound:
                    raise MMethodNotFound(
                        "The Command %s is not supported in %s plugin" %
                        (method_name, module_name))
            return CorePluginDefs.CORE_FAILURE, json.dumps(err_result)
        return CorePluginDefs.CORE_SUCCESS, json.dumps(result["result"])

    def execute_command(self, json_request, exec_opt):
        """Command execution point."""
        request = json.loads(json_request)
        module_name = request["module"]
        method_name = request["method"]
        fmt = self.EnumFormatType.Format_HumanReadable
        if "format" in request:
            fmt = request["format"]
        self._validateFormat(fmt)
        py_plugin_found = False
        if module_name in MPluginManager().get_plugin_list():
            plugin = MPluginManager().get_plugin(module_name)
            py_plugin_found = True
            if method_name in plugin.get_command_list():
                return self.get_python_command_metadata(
                    module_name, method_name, fmt)
        return self.get_cpp_command_metadata(module_name, method_name, fmt,
                                             py_plugin_found)


class CorePlugin(AbsNeoHostPluginIFC):
    """core plugin class extends AbsNeoHostPluginIFC"""
    def __init__(self):
        super(CorePlugin, self).__init__(
            CorePluginDefs.CORE_PLUGIN_NAME,
            CorePluginDefs.CORE_PLUGIN_DESC,
            CorePluginDefs.CORE_PLUGIN_VER)
        self.add_command(GetApplicationSupportedModulesCommand())
        self.add_command(GetApplicationModuleMetaDataCommand())
        self.add_command(GetApplicationMethodMetaDataCommand())
        self.add_command(SetApplicationTerminateJobCommand())
        self.add_command(GetApplicationFileMetaData())
        self.add_command(GetApplicationFileData())
