#!/usr/bin/python

# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# Python Imports

import logging
import os
import subprocess
import time
# Local Imports
from common_meta import MSingletonMeta
from neohost_common import NeoHostCommon
from neohost_logger import INIT

logger = logging.getLogger("neohost." + __name__)


class MCppProcess(object):
    """Singleton , holds the binary backend process objects, all binary
    backend requests are passed through here."""
    __metaclass__ = MSingletonMeta

    if NeoHostCommon.isWindowsOs():
        ENV_LD_LIBRARY_PATH = "PATH"
    else:
        ENV_LD_LIBRARY_PATH = "LD_LIBRARY_PATH"

    ENV_MCABLES_PATH = "MCABLES_PATH"

    def __init__(self):
        curr_dir = os.path.dirname(os.path.realpath(__file__))
        core_dir = "bin"
        if NeoHostCommon.development:
            core_dir = "src"
        
        if NeoHostCommon.isWindowsOs():
            ld_library_path = os.path.normpath(
              os.path.join(curr_dir, os.pardir, os.pardir, "core", core_dir))
            mcables_path = os.path.join(ld_library_path, "libmcables-1.dll")
        else:
            ld_library_path = os.path.normpath(
              os.path.join(curr_dir, os.pardir, os.pardir, "common", core_dir))
            mcables_path = os.path.join(ld_library_path, "mcables.so")
        prefix = os.environ.get(MCppProcess.ENV_LD_LIBRARY_PATH)
        if prefix:
            if NeoHostCommon.isWindowsOs():
                sep = ";"
            else:
                sep = ":"
            ld_library_path = prefix + sep + ld_library_path
        os.environ[MCppProcess.ENV_LD_LIBRARY_PATH] = ld_library_path

        os.environ[MCppProcess.ENV_MCABLES_PATH] = mcables_path

        exec_name = "neohost_core"
        if NeoHostCommon.isWindowsOs():
            exec_name = "neohost_core.exe"
        self.__core_exec = os.path.normpath(
            os.path.join(curr_dir, os.pardir, core_dir, exec_name))
        self._runProc()

    def _runProc(self):
        logger.log(INIT, "running cpp core process: %s", self.__core_exec)
        close_fds_val = True
        if NeoHostCommon.isWindowsOs():
            close_fds_val = False
        self.__core_binary_backend_process = subprocess.Popen(
            self.__core_exec,
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            universal_newlines=False,
            shell=True,
            close_fds=close_fds_val)
        time.sleep(0.5)
        if self.poll() is not None:
            out, err = self.__core_binary_backend_process.communicate()
            logger.error("Failed to run NeoHost backend process:\n"
                         "out:%s\nerr: %s", out, err)
            raise RuntimeError(
                "Failed to run NeoHost backend process, see log file for "
                "more details.")

    def poll(self):
        """Returns True if the backend process is still alive."""
        return self.__core_binary_backend_process.poll()

    def send_request(self, req_obj):
        """Posts a request to the backend binary process and returns the
        response."""
        # dispatch to cpp binary
        params = "%s %s %s %s\n" % (
            req_obj.module, req_obj.method, req_obj.exec_mode, req_obj.params)
        # make sure core backend process is still alive (if not, something
        # bad happened so exit)
        if self.poll() is not None:
            logger.error("NeoHost backend process terminated unexpectedly, "
                         "restarting process")
            out, err = self.__core_binary_backend_process.communicate()
            logger.error("NeoHost backend process terminated unexpectedly\n"
                         "out:%s\nerr: %s\nrestarting process", out, err)
            self._runProc()
        # post request
        self.__core_binary_backend_process.stdin.write(params)
        result = self.__core_binary_backend_process.stdout.readline()
        if self.poll() is not None and not NeoHostCommon.neohost_stopped:
            logger.error(
                "NeoHost backend process terminated unexpectedly: %s", result)
            raise RuntimeError(
                "NeoHost backend process terminated unexpectedly, "
                "see log file for more details")
        return result
