# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) 2016-2017, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --
import datetime
import json
import os
import tempfile

from common_meta import MEnumMeta
from core_plugin_defs import CorePluginDefs
from neohost_exceptions import MCorePluginError
from neohost_plugin_ifc import AbsNeoHostCommandIFC, EnumCmdType, \
    EnumCmdScope, EnumCmdExecMode
from neohost_common import NeoHostCommon


class EnumFileType(object):
    """File type enum."""
    __metaclass__ = MEnumMeta
    FT_Log = 1

    SupportedTypes = (FT_Log,)


class LogFileMetaData(object):
    def __init__(self):
        if NeoHostCommon.isWindowsOs():
            self.filePath = os.path.join(tempfile.gettempdir(), "neohost.log")
        else:
            self.filePath = "/var/log/neohost.log"
        self.sizeInBytes = 0
        self.numOfLines = 0
        self.maxLinesPerReq = 1000
        self.lastModified = ""

        self._checkFileValidity()
        self.numOfLines = self._getLogFileNumOfLines()

    def getData(self):
        return self.__dict__

    def _checkFileValidity(self):
        if not os.path.isfile(self.filePath):
            raise MCorePluginError("File Not Found: %s" % self.filePath)
        return True

    def _getLogFileNumOfLines(self):
        numOfLines = 0
        for _ in open(self.filePath):
            numOfLines += 1
        return numOfLines


class GetApplicationFileMetaData(AbsNeoHostCommandIFC):
    """Class that implements GetApplicationFileMetaData()
    NeoHost backend API command"""

    def __init__(self):
        super(GetApplicationFileMetaData, self).__init__(
            cmd_name="GetApplicationFileMetaData",
            cmd_desc="Get file metadata",
            cmd_type=EnumCmdType.Cmd_Type_Get,
            cmd_scope=EnumCmdScope.Cmd_Scope_Application,
            supp_exec_mask=EnumCmdExecMode.Cmd_Exec_Mode_All,
            cmd_in_desc="FileType: resource enum",
            cmd_out_desc="Specific per FileType",
            extra_str="")
        self._file_metadata_mapping = {
            EnumFileType.FT_Log: self._getLogFileMetadata
        }

    def execute_command(self, json_request, exec_opt):
        """
        Command execution point
        :param json_request:
        :param exec_opt:
        :return:
        """
        request = json.loads(json_request)
        fileType = request["fileType"]
        if fileType not in EnumFileType.SupportedTypes:
            raise MCorePluginError(
                "Unsupported file type: %s" % fileType)
        method = self._file_metadata_mapping.get(fileType)
        if not method:
            raise MCorePluginError(
                "Internal Error: could not find metdata for file type: %s" %
                fileType)
        return method()

    def _getLogFileMetadata(self):
        file_metadata = LogFileMetaData()

        mtime = os.path.getmtime(file_metadata.filePath)
        file_metadata.lastModified = str(
            datetime.datetime.fromtimestamp(mtime))
        file_metadata.sizeInBytes = os.path.getsize(file_metadata.filePath)
        return CorePluginDefs.CORE_SUCCESS, json.dumps(file_metadata.getData())


class GetApplicationFileData(AbsNeoHostCommandIFC):
    """Class that implements GetApplicationFileData()
    NeoHost backend API command"""

    def __init__(self):
        super(GetApplicationFileData, self).__init__(
            cmd_name="GetApplicationFileData",
            cmd_desc="Get file data",
            cmd_type=EnumCmdType.Cmd_Type_Get,
            cmd_scope=EnumCmdScope.Cmd_Scope_Application,
            supp_exec_mask=EnumCmdExecMode.Cmd_Exec_Mode_All,
            cmd_in_desc="fileType: resource enum, accessDataMod: object",
            cmd_out_desc="Specific per FileType",
            extra_str="")
        self._file_data_mapping = {
            EnumFileType.FT_Log: self._getLogFileData
        }

    def execute_command(self, json_request, exec_opt):
        """
        Command execution point
        :param json_request:
        :param exec_opt:
        :return:
        """
        request = json.loads(json_request)
        fileType = request["fileType"]

        # accessDataMod is not mandatory
        accessDataMod = request.get("accessDataMod", {})

        if fileType not in EnumFileType.SupportedTypes:
            raise MCorePluginError(
                "Unsupported file type: %s" % fileType)
        method = self._file_data_mapping.get(fileType)
        if not method:
            raise MCorePluginError(
                "Internal Error: could not find data for file type: %s" %
                fileType)
        return method(accessDataMod)

    def _getLogFileData(self, accessDataMod):
        file_metadata = LogFileMetaData()
        startLine = accessDataMod.get("startLine", 0)
        numOfLines = accessDataMod.get("numOfLines",
                                       file_metadata.maxLinesPerReq)
        if numOfLines > file_metadata.maxLinesPerReq:
            raise MCorePluginError("num of lines is bigger than: %s" %
                                   file_metadata.maxLinesPerReq)
        if numOfLines <= 0:
            raise MCorePluginError(
                "Invalid num of lines")

        try:
            with open(file_metadata.filePath) as fp:
                content = fp.readlines()
        except IOError:
            raise MCorePluginError("Failed to open file: %s",
                                   file_metadata.filePath)

        # if startLine is not valid as default give the last
        # maxLinesPerReq lines
        if startLine < 0:
            delta = len(content) + startLine
            startLine = max(0, delta)

        endLine = startLine + numOfLines
        req_content = content[startLine:endLine]
        result = json.dumps(req_content)
        del content
        del req_content
        return CorePluginDefs.CORE_SUCCESS, result
