#!/usr/bin/python

# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# Python Imports #####
import os
import time
import json
import logging
import threading
import tempfile

# local imports
from common_meta import MSingletonMeta
from conf_parser import MConfParser
import ctypes
from neohost_common import NeoHostCommon


logger = logging.getLogger("neohost." + __name__)


def getCurrentTimeStamp(formatted=True):
    currtime = time.time()
    if formatted:
        gtime = time.gmtime(currtime)
        currtime = time.strftime("%Y-%m-%dT%H:%M:%SZ", gtime)
    return currtime


class JobReporter(object):
    JOB_STAT_RUNNING = 0
    JOB_STAT_COMPLETE = 1
    JOB_STAT_TERMINATED = 2
    JOB_STAT_KILLED = 3

    JOB_STAT_MAPPPING = {
        JOB_STAT_RUNNING: "Running",
        JOB_STAT_COMPLETE: "Complete",
        JOB_STAT_TERMINATED: "Terminated",
        JOB_STAT_KILLED: "Killed",
    }

    def __init__(self, tid, outFileName):
        self._outFileName = outFileName
        self._msgIdx = 0
        self._jobStatus = JobReporter.JOB_STAT_RUNNING
        self._tid = tid

    @classmethod
    def jobStatusToString(cls, jobStatus):
        return cls.JOB_STAT_MAPPPING.get(jobStatus, "Unknown")

    def post(self, progress, extendedStr="", extendedJsonObjStr=None,
             finalResult="", resultOrErr=True):
        objValue = {}
        try:
            # write request to stream
            objValue["timestamp"] = getCurrentTimeStamp()
            objValue["msgIdx"] = self._msgIdx
            objValue["status"] = self.jobStatusToString(self._jobStatus)
            objValue["progress"] = progress
            objValue["extendedStr"] = extendedStr
            extendedObj = None
            if extendedJsonObjStr is not None:
                extendedObj = json.loads(extendedJsonObjStr)
            objValue["extendedObj"] = extendedObj
            if finalResult:
                attr = "result"
                if not resultOrErr:
                    attr = "error"
                objValue[attr] = json.loads(finalResult)
            msg = json.dumps(objValue)
            with open(self._outFileName, "a") as fp:
                fp.write(msg)
                fp.write(os.linesep)
        except Exception as e:
            logger.error(str(e))
            return False
        self._msgIdx += 1
        return True

    def setJobStatus(self, status):
        if status in JobReporter.JOB_STAT_MAPPPING:
            self._jobStatus = status

    def isDone(self):
        return self._jobStatus in (JobReporter.JOB_STAT_COMPLETE,
                                   JobReporter.JOB_STAT_KILLED,
                                   JobReporter.JOB_STAT_TERMINATED)

    def getThreadId(self):
        return self._tid


class ThreadInterrrupt(Exception):
    pass


# Singleton
class JobMgr(object):
    """JobMgr is a singleton that allows the management of python threads for Job execution"""
    __metaclass__ = MSingletonMeta

    def __init__(self):
        tmp_dir_path = MConfParser().get("path", "tmp_files")
        if NeoHostCommon.isWindowsOs():
            #Align with CPP core to get Tmp dir from the OS
            tmp_dir_path = tempfile.gettempdir()
        self.jobs_dir = os.path.join(tmp_dir_path, "jobs")
        self._startCleanupThread()
        self._reporters = {}

    def raiseThreadException(self, thread_id, exc_cls):
        ret = ctypes.pythonapi.PyThreadState_SetAsyncExc(
            ctypes.c_long(thread_id), ctypes.py_object(exc_cls()))
        if ret == 0:
            return False
        return True

    def generateJobToken(self, module, method, thread_id, timestamp):
        jobToken = "%s.%s.%s.%s" % (module, method, thread_id, timestamp)
        return jobToken

    def createJobReporter(self, tid, jobToken):
        logger.debug("adding job reporter: %s", jobToken)
        job_file = os.path.join(self.jobs_dir, jobToken)
        job_reporter = JobReporter(tid, job_file)
        self._reporters[jobToken] = job_reporter
        return job_reporter

    def removeJobReporter(self, job_token):
        logger.debug("removing job reporter: %s", job_token)
        if job_token in self._reporters:
            del self._reporters[job_token]

    def terminateJobs(self):
        for reporter in self._reporters.itervalues():
            reporter.setJobStatus(JobReporter.JOB_STAT_KILLED)
            reporter.post(-1, "Job killed!")

    def terminateJob(self, job_token):
        error = ""
        found = True
        if job_token not in self._reporters:
            found = False
        else:
            thread_id = self._reporters[job_token].getThreadId()
            if not self.raiseThreadException(thread_id, ThreadInterrrupt):
                error = "Failed to teminate Job: %s" % job_token
        return found, error

    def getActiveJobsCount(self):
        count = 0
        for reporter in self._reporters.itervalues():
            if not reporter.isDone():
                count += 1
        return count

    def _startCleanupThread(self):
        th = threading.Thread(target=self.cleanupTmpJobFiles)
        th.setDaemon(True)
        th.start()

    def cleanupTmpJobFiles(self):
        """Cleanup temporary files, used in a separate thread.
        No return value."""
        MAX_LIFETIME_IN_SEC = 3600

        if self.jobs_dir is None:
            return
        while True:
            logger.debug("cleanup thread wake up.")
            if os.path.isdir(self.jobs_dir):
                files = [f for f in os.listdir(self.jobs_dir) if
                         os.path.isfile(os.path.join(self.jobs_dir, f))]
                files = map(lambda f: os.path.join(self.jobs_dir, f), files)
                for tmpfile in files:
                    try:
                        file_stats = os.stat(tmpfile)
                        diff_in_sec = time.time() - file_stats.st_atime
                        # make sure more than MAX_LIFETIME_IN_SEC 10 minutes
                        # have passed
                        if (diff_in_sec > MAX_LIFETIME_IN_SEC):
                            logger.debug("cleanup thread removing file: %s",
                                         tmpfile)
                            os.remove(tmpfile)
                    except Exception as e:
                        logger.debug(
                            "cleanup thread exception caught: %s", str(e))
                        continue
            time.sleep(300)  # sleep 5 minutes
        return
