#!/usr/bin/python
# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


__version__ = '1.0'

try:
    # Python Imports #####
    import traceback
    import sys
    import os
    import signal
    import logging
    import threading
    import socket
    import optparse
    import tempfile

    # Local Imports #####
    from neohost_plugin_ifc import NeoHostBaseException
    from neohost_response import NeoHostResponse
    from request_mgr import RequestMgr
    from request_dispatcher import RequestDispatcher
    from job_mgr import JobMgr
    import neohost_logger
    from neohost_logger import INIT
    from neohost_common import NeoHostCommon
    import neohost_version
    from plugin_cmd_agent import PluginCmdAgent

except Exception as e:
    print("-E- could not import : %s" % str(e))
    traceback.print_exc()
    sys.exit(1)

####################
# initialize logger#
####################
logger = logging.getLogger("neohost")

if NeoHostCommon.isWindowsOs():
    log_file_path = os.path.join(tempfile.gettempdir(), "neohost.log")
    neohost_logger.setLogger(log_file_path)
else:
    import readline  # noqa
    neohost_logger.setLogger("/var/log/neohost.log")

global neohostApp
neohostApp = None


######################################################################
# Description:  Signal handler
######################################################################
def signal_handler_(sig, frame):
    """Signal Handling function.
        the function does not return."""
    logger.log(INIT, "Signal %d Received, Exiting..." % sig)
    NeoHostCommon.neohost_stopped = True
    if neohostApp:
        neohostApp.cleanup()
    sys.exit(2)


class NeoHostApp(object):
    """Main class, in charge of listening to user request,
    validating input and processing the request."""

    def __init__(self, port, prompt):
        self._port = port
        self._prompt = prompt
        self._socket = None
        self._socket_closed = True
        self._connections = {}
        self._conn_mutex = threading.Lock()
        self._req_dispatcher = RequestDispatcher()
        self._req_mgr = RequestMgr(self._req_dispatcher)
        self._cmd_agent = PluginCmdAgent(self._req_dispatcher)
        self._job_mgr = JobMgr()

    def request_loop(self):
        """REPL - continuously get requests from user,
        process them and print result."""
        try:
            while True:
                request = raw_input(self._prompt)
                response = self._req_mgr.requestRecieved(request)
                if response:
                    print response
                sys.stdout.flush()
        except EOFError:
            NeoHostCommon.neohost_stopped = True
            logger.info("request loop done - calling cleanup")
            self.cleanup()

    def run(self):
        """Performs Initializations and executes the request loop."""
        # event loop
        self._req_mgr.start()

        logger.info("run NeoHost main loop")
        if self._port:
            th = threading.Thread(target=self._listen, args=(self._port,))
            th.start()

        self.request_loop()

    def _handleTcpConnection(self, conn, addr):
        logger.info("handling tcp connection for: %s", addr)
        BUFFER_SIZE = 1024
        if self._prompt:
            conn.send(self._prompt)

        while True:
            try:
                msg_ready = False
                request = ""
                while not msg_ready:
                    request += conn.recv(BUFFER_SIZE)
                    if len(request) < BUFFER_SIZE or request.endswith("\n"):
                        msg_ready = True
            except socket.timeout:
                if not self._socket_closed:
                    continue
                break
            except:
                break
            if not request or NeoHostCommon.neohost_stopped:
                break
            response = self._req_mgr.requestRecieved(request)
            if response is None:
                break
            conn.send(response + "\n")
            if self._prompt:
                conn.send(self._prompt)
        self._conn_mutex.acquire()
        if conn in self._connections:
            try:
                addr = self._connections.pop(conn)
                logger.info("Closed connection: %s", addr)
                conn.shutdown(2)
                conn.close()
            except:
                pass
        self._conn_mutex.release()
        logger.info("TCP connection handling is done")

    def _listen(self, port):
        self._socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self._socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        self._socket_closed = False
        try:
            self._socket.bind(("127.0.0.1", port))
            self._socket.listen(5)
        except socket.error as e:
            print "-E- Failed to open socket:", str(e)
            self._socket.close()
            self._socket = None
            os.kill(os.getpid(), signal.SIGINT)
            return

        logger.info("listening on port: %s", port)
        while True:
            try:
                conn, addr = self._socket.accept()
                if NeoHostCommon.neohost_stopped:
                    break
                logger.info("got tcp connection: %s", addr)
            except socket.error:
                if self._socket_closed:
                    break
                continue
            self._conn_mutex.acquire()
            self._connections[conn] = addr
            self._conn_mutex.release()
            th = threading.Thread(
                target=self._handleTcpConnection, args=(conn, addr))
            th.start()

    def cleanup(self):
        logger.log(INIT, "NeoHost cleanup")
        self._req_mgr.stop()
        self._job_mgr.terminateJobs()
        if self._socket:
            logger.log(INIT, "NeoHost socket cleanup")
            self._conn_mutex.acquire()
            try:
                logger.info("Closing socket...")
                self._socket.shutdown(2)
                self._socket.close()
                for conn, addr in self._connections.iteritems():
                    logger.info("Closing connection: %s", addr)
                    try:
                        conn.shutdown(2)
                        conn.close()
                    except:
                        pass
                self._connections.clear()
                self._socket_closed = True
            finally:
                self._conn_mutex.release()
        logger.info("NeoHost cleanup done!")

    @classmethod
    def parseArgs(cls):
        parser = optparse.OptionParser(prog="neohost")
        parser.add_option(
            "-p", "--prompt", action="store_true", help="add prompt")
        parser.add_option(
            "-v", "--version", action="store_true",
            help="show program's version number and exit")
        parser.add_option(
            "--development", action="store_true", help=optparse.SUPPRESS_HELP)
        parser.add_option(
            "-c", "--command-help", action="store_true",
            help="show command syntax and exit")
        parser.add_option(
            "-s", "--socket", type=int, help="use tcp socket")
        parser.add_option(
            "-l", "--log-level", choices=["DEBUG", "INFO", "WARNING", "ERROR"],
            help="logging level")
        options, _ = parser.parse_args()
        return options


if __name__ == '__main__':
    try:
        args = NeoHostApp.parseArgs()
        if args.version:
            neohost_version.printVersionString()
            sys.exit(0)
        if args.command_help:
            print RequestDispatcher.get_cmd_format()
            sys.exit(0)
        if args.development:
            NeoHostCommon.development = True
        neohost_logger.updateLogger(args.log_level)
        prompt = ""
        if args.prompt:
            prompt = "[NeoHost]# "
        signal.signal(signal.SIGINT, signal_handler_)
        logger.log(INIT, "Neohost start")
        neohostApp = NeoHostApp(args.socket, prompt)
        neohostApp.run()

    except Exception as e:
        exception_tb = traceback.format_exc()
        logger.error(exception_tb)
        exc = NeoHostBaseException(str(e), "core")
        error = exc.to_dict()
        resp = NeoHostResponse()
        resp.fromResult(1, error)
        print resp.dump()
        sys.exit(1)
    sys.exit(0)
