# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) 2016-2017, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --
# Created on Nov 3, 2016
#
# @author: samerd

import logging
import logging.handlers
from conf_parser import MConfParser
from neohost_common import NeoHostCommon
from neohost_exceptions import MException

INIT = 51

logging.addLevelName(INIT, 'INIT')


class LoggingPropertiesMgr(object):
    logger = logging.getLogger("neohost")

    def __init__(self):
        self.file_name = MConfParser().get("path", "logging_properties")
        self.content = []

    def _readProperies(self):
        del self.content[:]
        with open(self.file_name, "r") as fp:
            for line in fp:
                self.content.append(line)

    def _writeProperties(self):
        if not self.content:
            return
        with open(self.file_name, "w") as fp:
            for line in self.content:
                fp.write(line)

    def _findLoggingLevel(self):
        line_num = -1
        logging_level = None
        for line in self.content:
            line_num += 1
            line = line.strip()
            if not line or line.startswith("#"):
                continue
            if "=" not in line:
                continue
            attr, val = line.split("=")
            attr = attr.strip()
            val = val.strip()
            if attr == "level":
                logging_level = val.split(",")[0]
                break
        return line_num, logging_level

    def getLoggingLevel(self):
        logging_level = None
        try:
            self._readProperies()
        except Exception as e:
            self.logger.error("Get Error while reading from file %s: %s",
                              self.file_name, str(e))
            return logging_level
        _, logging_level = self._findLoggingLevel()
        if logging_level is None:
            self.logger.error("could not find logging level in file %s",
                              self.file_name)
        if logging_level == 'WARN':
            logging_level = 'WARNING'
        return logging_level

    def setLoggingLevel(self, log_level):
        if not self.content:
            try:
                self._readProperies()
            except Exception as e:
                self.logger.error("Get Error while reading from file %s: %s",
                                  self.file_name, str(e))
                return
        line_num, exist_log_level = self._findLoggingLevel()
        if not exist_log_level:
            self.logger.error("Failed to set logging level in file %s",
                              self.file_name)
        line = self.content[line_num]
        if log_level == 'WARNING':
            log_level = 'WARN'
        self.content[line_num] = line.replace(exist_log_level, log_level)
        try:
            self._writeProperties()
        except Exception as e:
            self.logger.error("Get Error while writing from file %s: %s",
                              self.file_name, str(e))
            return


def setLogger(filename):
    _setFileHandler(filename)
    _setSyslogHandler()


def _setFileHandler(filename):
    logger = logging.getLogger("neohost")
    FORMAT = '%(asctime)s %(thread)s %(name)s [%(levelname)s] %(message)s'
    fmt = logging.Formatter(FORMAT)
    hdlr = logging.FileHandler(filename)
    hdlr.setFormatter(fmt)
    logger.addHandler(hdlr)


def _setSyslogHandler():
    logger_obj = logging.getLogger("neohost")
    conf_parser_obj = MConfParser()
    syslog_enabled = conf_parser_obj.getboolean("logging", "syslog_enable")
    if not syslog_enabled:
        return
    if NeoHostCommon.isWindowsOs():
        raise MException("SysLog is not supported in Windows, disable this option in the neohost_core.ini")
    syslog_addr = conf_parser_obj.get("logging", "syslog_addr")
    syslog_level = conf_parser_obj.get("logging", "syslog_level")
    logging_level = logging.getLevelName(syslog_level)
    if syslog_addr:
        handler = logging.handlers.SysLogHandler(syslog_addr)
        handler.setFormatter(logging.Formatter(
            "%(name)s[%(process)d]: %(levelname)s - %(message)s"))
        handler.setLevel(logging_level)
        logger_obj.addHandler(handler)


def updateLogger(logging_level):
    logger = logging.getLogger("neohost")
    log_props = LoggingPropertiesMgr()
    existing_logging_level = log_props.getLoggingLevel()
    if not logging_level:
        set_logging_level = existing_logging_level
    else:
        set_logging_level = logging_level
    if set_logging_level:
        logger.setLevel(logging.getLevelName(set_logging_level))

    if logging_level and logging_level != existing_logging_level:
        log_props.setLoggingLevel(logging_level)
