#!/usr/bin/python

# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# Python Imports

from common_meta import MEnumMeta
import abc
import json


class EnumCmdType(object):
    """Command type enum."""
    __metaclass__ = MEnumMeta
    Cmd_Type_Get = 0
    Cmd_Type_Set = 1


class EnumCmdScope(object):
    """Command scope enum."""
    __metaclass__ = MEnumMeta
    Cmd_Scope_Application = 0
    Cmd_Scope_System = 1
    Cmd_Scope_Device = 2
    Cmd_Scope_Port = 3
    Cmd_Scope_Interface = 4


class EnumCmdExecMode(object):
    """Command execution mode enum."""
    __metaclass__ = MEnumMeta
    Cmd_Exec_Mode_Sync = 1 << 0
    Cmd_Exec_Mode_ASync = 1 << 1
    Cmd_Exec_Mode_All = Cmd_Exec_Mode_Sync | Cmd_Exec_Mode_ASync


class NeoHostExecOpt(object):
    """Additional execution option class, this will be passed to a command
    object execute_command() method
    intended to provide execution information, expanded in the future for
    loggin and async reporting services"""
    def __init__(self, exec_mode):
        self.exec_mode = exec_mode
        self.job_reporter = None


class NeoHostBaseException(Exception):
    """NeoHost base exception, a plugin module should throw and instance of
    this exception to the framework in case throwing is needed.
    the Ctor conforms to the layout of an error object as defined in the
    API doc"""
    def __init__(self, msg_str, source_str, error_code=1):
        self.__error_code = error_code
        self.__msg = msg_str
        self.__source = source_str

    def to_dict(self):
        error_dict = dict(code=1, message="N/A", source="N/A")
        # check types for safety
        if isinstance(self.__error_code, int):
            error_dict["code"] = self.__error_code
        if isinstance(self.__msg, basestring):
            error_dict["message"] = self.__msg
        if isinstance(self.__source, basestring):
            error_dict["source"] = self.__source
        return error_dict

    def to_json_str(self):
        return json.dumps(self.to_dict())

    def __str__(self):
        return str(self.__msg)


class AbsNeoHostPluginIFC(object):
    """Class, for plugins to either instantiate or inherit and extend."""

    def __init__(self, plugin_name, plugin_desc, plugin_ver, cmd_obj_vec=[]):
        """Ctor:
        plugin_name: string depicting the plugin name
        plugin_desc: string depicting the plugin description
        plugin_ver: tuple of (major,minor) version of the plugin
        cmd_obj_vec: vector of command objects"""
        self.__plugin_name = plugin_name
        self.__plugin_desc = plugin_desc
        self.__plugin_ver = plugin_ver
        self.__cmd_dict = dict()
        for cmd in cmd_obj_vec:
            self.add_command(cmd)

    def get_command_list(self):
        """retireve a string list of available commands"""
        return self.__cmd_dict.keys()

    def get_command(self, cmd_name):
        """get cmd_name command object."""
        if cmd_name not in self.__cmd_dict:
            raise NeoHostBaseException("command %s not found." % cmd_name,
                                       "coreIFC")
        return self.__cmd_dict[cmd_name]()

    def add_command(self, cmd_obj):
        """Add command: cmd_obj to plugin"""
        if isinstance(cmd_obj, AbsNeoHostCommandIFC):
            self.__cmd_dict[cmd_obj.get_command_name()] = cmd_obj.__class__
        else:
            raise NeoHostBaseException(
                "command object does not realize: AbsNeoHostCommandIFC .",
                "coreIFC")

    def get_plugin_name(self):
        """Get the plugin name."""
        return self.__plugin_name

    def get_plugin_description(self):
        """Get the plugin description."""
        return self.__plugin_desc

    def get_plugin_version(self):
        """Get the plugin version."""
        return self.__plugin_ver


class AbsNeoHostCommandIFC(object):
    """Abstarct class for implementing commands"""
    __metaclass__ = abc.ABCMeta

    def __init__(self, cmd_name, cmd_type, cmd_scope, supp_exec_mask,
                 cmd_desc, cmd_in_desc, cmd_out_desc, extra_str="", **kwargs):
        """Ctor:
        cmd_name: string depicting name of the command.
        cmd_type: an EnumCmdType Value.
        cmd_scope: an EnumCmdScope value.
        supp_exec_mask: supported execution mask, bitwise or'ed of
                        EnumCmdExecMode values.
        cmd_desc: command description string.
        cmd_in_desc: command input description string.
        cmd_out_desc: command output description string.
        extra_str: extra information string."""
        self.__cmd_name = cmd_name
        self.__cmd_type = cmd_type
        self.__cmd_scope = cmd_scope
        self.__supp_exec_mask = supp_exec_mask
        self.__cmd_desc = cmd_desc
        self.__cmd_in_desc = cmd_in_desc
        self.__cmd_out_desc = cmd_out_desc
        self.__extra_str = extra_str

    @abc.abstractmethod
    def execute_command(self, json_request, exec_opt):
        """execute a command, json_request is a JSON string.
        exec_opt is an NeoHostExecOpt object
        Returns a tuple of (rc, json_response_str)"""
        pass

    def get_command_cli_metadata(self):
        """Return a dictionary of the command's CLI metata data as described
        in the API document
        overload it if its desired to be exposed as a CLI command
        (in the NeoHost CLI application)."""
        return None

    def get_command_gui_metadata(self):
        """Return a dictionary of the command's GUI metata data specifically
        tailored for the NeoHost GUI application."""
        return None

    def get_command_type(self):
        """Returns the command type(EnumCmdType)."""
        return self.__cmd_type

    def get_command_scope(self):
        """Returns the command scope(EnumCmdScope)."""
        return self.__cmd_scope

    def get_command_supp_exec_modes(self):
        """Returns the command's supported execution modes."""
        return self.__supp_exec_mask

    def get_command_name(self):
        """Returns the command name."""
        return self.__cmd_name

    def get_command_desc(self):
        """Returns the command description."""
        return self.__cmd_desc

    def get_command_input_desc(self):
        """Returns the command input description."""
        return self.__cmd_in_desc

    def get_command_output_desc(self):
        """Returns the command output description."""
        return self.__cmd_out_desc

    def get_command_extra_str(self):
        """Returns the command extra string."""
        return self.__extra_str
