#!/usr/bin/python

# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# Python Imports
import sys
import os
import logging
# Local imports
import neohost_exceptions
from common_meta import MSingletonMeta
from path_provider import MPathProvider
from conf_parser import MConfParser
from neohost_common import NeoHostCommon

####################
# initialize logger#
####################
logger = logging.getLogger("neohost." + __name__)


class MPluginPathFinder(object):
    """MPluginPathFinder class searches for available plugins and retrieves
    their paths. this is a service class used by MSchemaValidator and
    MPluginManager"""
    def __init__(self):
        # initialize needed paths
        self.__neohost_root_dir = MPathProvider().get_root_dir()
        self._plugin_search_paths = [
            self.__neohost_root_dir,
            os.path.join(self.__neohost_root_dir, MPathProvider.PLUGINS_DIR)]
        self._conf_dict = None
        conf_parser = MConfParser()
        pluginSearchPath = conf_parser.get("plugin", "search_path")
        if not pluginSearchPath:
            # TODO: append root dir path so plugins can import
            #        neohost_plugin_ifc.py
            self._plugin_search_paths.extend(pluginSearchPath)

    def get_plugin_path(self, plugin_name):
        """Get a specific plugin path, returns the plugin path if found.
        In case of failure raise exception"""
        for possible_path in self._plugin_search_paths:
            logger.debug("searching for plugin: %s in %s" %
                         (plugin_name, possible_path))
            ppath = os.path.join(possible_path, plugin_name)
            if os.path.exists(ppath):
                logger.debug("plugin: %s located successfully." % plugin_name)
                return ppath
        raise neohost_exceptions.MPathError(
            "Failed to locate %s plugin path" % plugin_name)

    def get_plugin_search_paths(self):
        """Returns the plugin search paths"""
        return self._plugin_search_paths


class MPluginManager(object):
    """MPluginManager class manages the available NeoHost python plugins in
    the system. this class is a Singleton which is in charge of plugin object
    initialization and retrieval."""
    __metaclass__ = MSingletonMeta

    class MPluginContext(object):
        """Plugin context class, hold the module loaded and the plugin object
        it provides"""
        def __init__(self, plugin_obj, module=None):
            self.module = module
            self.plugin_obj = plugin_obj

    def __init__(self):
        self.__plugin_path_finder = MPluginPathFinder()
        self.__plugin_search_paths = \
            self.__plugin_path_finder.get_plugin_search_paths()
        self.__plugins_dict = {}
        # update system path
        map(sys.path.append,  self.__plugin_search_paths)
        self.__local_plugin_list = []

    def get_plugin_list(self):
        """Returns the supported python plugins"""
        if NeoHostCommon.development:
            init_name = "__init__.py"
        else:
            init_name = "__init__.pyo"
        plugin_list = []
        for path_str in self.__plugin_search_paths:
            # list all dirs
            for plugin_candidate in os.listdir(path_str):
                full_path = os.path.join(path_str, plugin_candidate)
                if os.path.isdir(full_path):
                    if os.path.exists(os.path.join(full_path, init_name)):
                        # bullseye
                        plugin_name = os.path.basename(
                                    os.path.normpath(plugin_candidate))
                        if not(plugin_candidate in plugin_list) and \
                           plugin_name != "plugins":
                            plugin_list.append(plugin_name)
        plugin_list.extend(self.__local_plugin_list)
        return list(set(plugin_list))

    def get_plugin_full_path(self, plugin_name):
        """
        :param plugin_name:
        :return:  returns the full path of a plugin
        """
        for path_str in self.__plugin_search_paths:
            # list all dirs
            if plugin_name in os.listdir(path_str):
                full_path = os.path.join(path_str, plugin_name)
                if os.path.isdir(full_path):
                    if os.path.exists(os.path.join(full_path, "__init__.py")):
                        # bullseye
                        return full_path
        return None

    def get_plugin(self, plugin_name):
        """Returns a specific plugin object if found. Upon first query the
        module is loaded and initalized.
        Raises an exception in case of failure."""
        if plugin_name not in self.__plugins_dict:
            try:
                # sys.path.append(self.get_plugin_full_path(plugin_name))
                module = __import__(plugin_name, globals(), locals(), [], -1)
                plugin_obj = module.class_factory()
                self.__plugins_dict[plugin_name] = \
                    MPluginManager.MPluginContext(plugin_obj, module)
            except Exception as e:
                raise neohost_exceptions.MPluginManagerError(
                    "failed to load plugin: %s.%s" % (plugin_name, str(e)))
        return self.__plugins_dict[plugin_name].plugin_obj

    def add_local_plugin(self, plugin_obj):
        """Adds a local plugin to the instance(i.e the builtin core plugin)"""
        plugin_name = plugin_obj.get_plugin_name()
        if plugin_name not in self.__plugins_dict:
            self.__local_plugin_list.append(plugin_name)
            self.__plugins_dict[plugin_name] = \
                MPluginManager.MPluginContext(plugin_obj)
