# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) 2016-2017, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --
'''
Created on Nov 2, 2016

@author: samerd
'''
# Python Imports #####
import time
import json
import logging
import thread

# Local Imports #####
import neohost_exceptions
from neohost_common import NeoHostCommon
from plugin_manager import MPluginManager
from neohost_plugin_ifc import NeoHostExecOpt, \
    NeoHostBaseException, EnumCmdExecMode
from job_mgr import JobMgr, JobReporter, ThreadInterrrupt
from neohost_response import NeoHostResponse

logger = logging.getLogger("neohost." + __name__)


class PythonProcessor(object):

    MAX_ASYNC_COMMANDS = 32

    def __init__(self):
        self._plugin_mngr = MPluginManager()
        self._job_mgr = JobMgr()

    def process_command(self, req_obj):
        """Sends a request for execution by the appropriate python command
        object.
        Return a JSON string response."""
        try:
            logger.info("Processing request: module: %s, method: %s",
                        req_obj.module, req_obj.method)
            plugin_obj = self._plugin_mngr.get_plugin(req_obj.module)
            # Check plugin version
            p_ver = plugin_obj.get_plugin_version()
            NeoHostCommon.check_python_plugin_version(p_ver[0], p_ver[1])
            # Get command object
            command_obj = plugin_obj.get_command(req_obj.method)
            # Check if execution mode is supported
            if (command_obj.get_command_supp_exec_modes() &
                    req_obj.exec_mode) == 0:
                raise neohost_exceptions.MException(
                    "Unsupported execution mode (%d) for command: %s.%s" %
                    (req_obj.exec_mode, req_obj.module, req_obj.method))
            exec_opt = NeoHostExecOpt(exec_mode=req_obj.exec_mode)
            (rc, response) = self.exec_command(
                command_obj, req_obj.module, req_obj.method, req_obj.params,
                exec_opt)
        except NeoHostBaseException as e:
            rc = 1
            response = e.to_dict()
        except Exception as e:
            rc = 1
            exc = NeoHostBaseException(str(e), req_obj.module)
            response = exc.to_dict()
        resp_obj = NeoHostResponse()
        resp_obj.fromResult(rc, response)
        return resp_obj.getData()

    def exec_command_async(self, command_obj, module_name, method_name,
                           params_str, exec_opt, timestamp):
        tid = thread.get_ident()
        jobToken = self._job_mgr.generateJobToken(
             module_name, method_name, tid, timestamp)
        exec_opt.job_reporter = self._job_mgr.createJobReporter(tid, jobToken)
        exec_opt.job_reporter.post(0, "Job Started!")
        ret = False
        try:
            _, result = command_obj.execute_command(params_str, exec_opt)
            message = "Operation completed."
            ret = True
        except NeoHostBaseException as e:
            result = e.to_json_str()
            message = "Operation completed with errors."
        except ThreadInterrrupt as e:
            exec_opt.job_reporter.setJobStatus(JobReporter.JOB_STAT_TERMINATED)
            exec_opt.job_reporter.post(-1, "Job Terminated!")
            self._job_mgr.removeJobReporter(jobToken)
            return
        except Exception as e:
            exc = NeoHostBaseException(str(e), module_name)
            result = exc.to_json_str()
            message = "Operation completed with errors."
        exec_opt.job_reporter.setJobStatus(JobReporter.JOB_STAT_COMPLETE)
        exec_opt.job_reporter.post(100, message, None, result, ret)
        self._job_mgr.removeJobReporter(jobToken)

    def exec_command(self, command_obj, module_name, method_name, params_str,
                     exec_opt):
        if exec_opt.exec_mode == EnumCmdExecMode.Cmd_Exec_Mode_Sync:
            res = command_obj.execute_command(params_str, exec_opt)
            return res
        current_jobs = self._job_mgr.getActiveJobsCount()
        if current_jobs >= self.MAX_ASYNC_COMMANDS:
            raise neohost_exceptions.MException(
                    "Number of running async jobs exceeds its limit: %s" %
                    current_jobs)
        timestamp = time.time()
        tid = thread.start_new_thread(
            self.exec_command_async,
            (command_obj, module_name, method_name, params_str, exec_opt,
             timestamp))
        job_token = self._job_mgr.generateJobToken(
             module_name, method_name, tid, timestamp)
        res = {
            "jobToken": job_token
        }
        rc = 0
        return rc, json.dumps(res)
