#!/usr/bin/python
# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# Python Imports #####
import logging
import json

# Local Imports #####
import neohost_exceptions
import neohost_schema
from plugin_manager import MPluginManager
from cpp_process import MCppProcess
from core_plugin import CorePlugin
from neohost_request import NeoHostRequest
from neohost_response import NeoHostResponse
from python_processor import PythonProcessor
from neohost_exceptions import MInvalidRequest
from neohost_common import NeoHostCommon
from neohost_plugin_ifc import EnumCmdExecMode


logger = logging.getLogger("neohost." + __name__)


class RequestDispatcher(object):
    """Main class, in charge of listening to user request,
    validating input and processing the request."""

    def __init__(self, *args, **kwargs):
        self._path_finder = neohost_schema.MPluginPathFinder()
        self._schema_validator = neohost_schema.MSchemaValidator()
        logger.debug("schema validator initiated")
        self._plugin_mngr = MPluginManager()
        # inject local plugins
        self._plugin_mngr.add_local_plugin(CorePlugin())
        logger.debug("plugin manager initiated")
        self._core_cpp_process = MCppProcess()
        self._python_processor = PythonProcessor()

    @classmethod
    def get_cmd_format(cls):
        return "Neohost Command Format: %s" % NeoHostRequest.get_req_struct()

    def is_python_command(self, module_name, method_name):
        """Returns True if a method: 'method_name' is available within the
        module: 'module_name'."""
        if module_name in self._plugin_mngr.get_plugin_list():
            plugin = self._plugin_mngr.get_plugin(module_name)
            if method_name in plugin.get_command_list():
                return True
        return False

    def dispatch_python_command(self, req_obj):
        """Sends a request for execution by the appropriate python command
        object.
        Return a JSON string response."""
        return self._python_processor.process_command(req_obj)

    def dispatch_cpp_binary_command(self, req_obj):
        """Sends a request for execution by passing the request to the binary
        process.
        Return a JSON string response."""
        # dispatch to cpp binary
        return self._core_cpp_process.send_request(req_obj)

    def dispatch_request(self, req_obj):
        # check if its a python plugin
        if self.is_python_command(req_obj.module, req_obj.method):
            # dispatch request to python plugin
            return self.dispatch_python_command(req_obj)
        else:
            return self.dispatch_cpp_binary_command(req_obj)

    def _validate_json(self, request_str):
        try:
            request = json.loads(request_str)
        except ValueError as e:
            logger.warning("Got request with invalid JSON: '%s'", request_str)
            raise neohost_exceptions.MParsingError(
                "Failed to parse request as json: %s" % str(e))
        return request

    def process_request(self, request_str):
        """Process a single user request.
            Returns a JSON string response."""
        msg_id = None
        response = None
        try:
            request_str = request_str.strip()
            if not request_str:
                logger.debug("Got empty request")
                raise MInvalidRequest(self.get_cmd_format())
            # validate agains basic request schema
            logger.debug("processing request: %s" % request_str)
            logger.debug("processing request JSON")
            request = self._validate_json(request_str)

            # if exec_mode is missing add the default value
            request.setdefault(NeoHostRequest.EXEC_MODE,
                               EnumCmdExecMode.Cmd_Exec_Mode_Sync)

            msg_id = request.get(NeoHostRequest.ID)
            logger.debug("validating request against basic schema")
            request = self._schema_validator.validate_basic_request(request)

            # validate against plugin specific schema
            logger.debug(
                "validating request params against plugin specific schema")
            # TODO - check why this function do not replace request with return value. doing so causing an error
            self._schema_validator.validate_request_params(request)

            # call core binary
            logger.debug("executing command")
            req_obj = NeoHostRequest(request)

            # dispatch request
            response_str = self.dispatch_request(req_obj)
            if NeoHostCommon.neohost_stopped:
                return ""
            # validate response
            logger.debug("validating response against plugin specific schema")
            response = self._schema_validator.validate_response(
                request, response_str)

            # TODO: if the request is GetApplicationJobStatus ,
            #       if one of the messages in the response
            #       contain the final result, we should validate it against the
            #       specific schema
            # assemble result

            resp_obj = NeoHostResponse(response)
        except neohost_exceptions.MException as e:
            resp_obj = NeoHostResponse()
            error_dict = e.to_dict()
            resp_obj.fromResult(1, error_dict)
        except Exception as e:
            # traceback.print_exc()
            exc = neohost_exceptions.NeoHostBaseException(str(e),
                                                          "core")
            error = exc.to_dict()
            resp_obj = NeoHostResponse()
            resp_obj.fromResult(1, error)
        resp_obj.setID(msg_id)
        response = resp_obj.dump()
        logger.debug("finished processing request - response: %s" % response)
        return response
