#!/usr/bin/python
# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Ariel Gaisinsky
# @date: Dec 10, 2017


import json
import logging

from cmd_exec import Command
from config_flows_exceptions import ConfigFlowsException
from config_flows_defs import ConfigFlowsDefs
from neohost_common import NeoHostCommon
from request_factory import RequestFactory
from msg_mgr import MsgMgr, MsgConstants


class ConfigFlowsCommon:

    @classmethod
    def raiseException(self, message):
        raise ConfigFlowsException(message)


    @classmethod
    def checkDriverVersion(self, logger):
        if not NeoHostCommon.isWindowsOs():
            ConfigFlowsCommon.checkOfedInstalled(logger, True)
        else:
            ConfigFlowsCommon.checkWinOFVersion()


    @classmethod
    def commandExecute(self, cmd, failMsg=None, raiseExcept=True):
        rc, out, err = Command(cmd).execute()
        if failMsg is None:
            failMsg = "Command %s has failed." % cmd
        if out == "":
            out = err
        if raiseExcept and rc:
            raise ConfigFlowsException("%s. %s" % (failMsg, out))
        return out

    @classmethod
    def getDeviceInterfaces(self, devUid, logger):
        if not NeoHostCommon.isWindowsOs():
            return ConfigFlowsCommon.getDeviceInterfacesLinux(devUid)
        else:
            return ConfigFlowsCommon.getDeviceInterfacesWindows(devUid, logger)


    @classmethod
    def getDeviceInterfacesLinux(self, devUid):
        ###
        sub_req_obj = RequestFactory.createRequest(
            1, "mftCore", "GetSystemDevices", None)
        _, rc, response = MsgMgr().handle(
            MsgConstants.NEOHOST_REQUEST, sub_req_obj)
        if rc != 0:
            raise ConfigFlowsException(
                "Failed to retrieve system devices: %s" %
                response["message"])

        ifc_list = []
        device_found = False
        for curr_dev_data in response:
            if curr_dev_data["uid"] == devUid:
                for port in curr_dev_data["ports"]:
                    for pf in port["physicalFunctions"]:
                        for ifc in pf["networkInterfaces"]:
                            ifc_list.append(ifc.strip())
                        device_found = True

        if not device_found:
            raise ConfigFlowsException(
                "Failed to find device %s" % devUid)

        if not ifc_list:
            raise ConfigFlowsException(
                "Failed to find network interfaces for device %s" % devUid)

        return ifc_list


    @classmethod
    def convertInterfaceToMlxDev(self, ifc_id):
        cmd = "ibdev2netdev | grep -w %s | cut -d \' \' -f1" % ifc_id
        failMsg = "Could not convert interface to mlx_dev for %s. " \
                  "Command -%s- has failed." % (ifc_id, cmd)
        out = ConfigFlowsCommon.commandExecute(cmd, failMsg)
        return out.rstrip('\r\n')


    @classmethod
    def preValidateCommand(self, cmd_to_validate, logger):
        logger.debug("__preValidateCommand")
        cmd = "which %s" % cmd_to_validate
        failMsg = "Could not set configuration. Command -%s- not found." % cmd_to_validate
        ConfigFlowsCommon.commandExecute(cmd, failMsg)


    @classmethod
    def checkIfIfcEthernet(self, ifc_id, logger):
        logger.debug("checking if Ethernet interface.")
        if not ConfigFlowsCommon.isEthernetInterface(ifc_id):
            raise ConfigFlowsException("Could not set configuration. Configuration supported only for Ethernet.")


    @classmethod
    def isEthernetInterface(self, ifc_id):
        cmd = "ip addr show %s | grep ether" % ifc_id
        rc, _, _ = Command(cmd).execute()
        if rc:
            return False
        return True


    @classmethod
    def checkOfedInstalled(self, logger, needed_ver_check=False):
        logger.debug("checking if OFED installed")
        cmd = "ofed_info -s"
        failMsg = "OFED should be installed."
        out = ConfigFlowsCommon.commandExecute(cmd, failMsg)

        if needed_ver_check:
            ver_num_parts = out.split("-")
            parts_len = len(ver_num_parts)
            if parts_len<2:
                raise ConfigFlowsException(
                    "Cannot compare OFED version - %s" % out)

            ver_num = "%s-%s" % (ver_num_parts[-2], ver_num_parts[-1])
            if ConfigFlowsDefs.OFED_MIN_VERSION > ver_num:
                raise ConfigFlowsException(
                    "OFED version %s is smaller than %s which needed for  "
                    "configuration." % (ver_num, ConfigFlowsDefs.OFED_MIN_VERSION))


    @classmethod
    def checkInterfaceExist(self, ifc_id, logger):
        logger.debug("Checking if Interface Exist.")
        cmd = "ip addr show %s" % ifc_id
        failMsg = "interface -%s- unknown." % ifc_id
        ConfigFlowsCommon.commandExecute(cmd, failMsg)


    @classmethod
    def checkFWVersion(self, mlx_dev_name, logger):

        logger.debug("Checking FW version.")
        cmd = "cat /sys/class/infiniband/%s/fw_ver" % mlx_dev_name
        failMsg = "could not find FW version by - %s command." % cmd
        out = ConfigFlowsCommon.commandExecute(cmd, failMsg)

        ver_parts = out.split(".")
        if len(ver_parts) != 3:
            raise ConfigFlowsException("Cannot compare FW version - %s" % out)
        relevant_ver = "%s.%s" % (ver_parts[1], ver_parts[2])

        if ConfigFlowsDefs.FW_MIN_VERSION > relevant_ver:
            raise ConfigFlowsException(
                "FW version %s is older than %s which needed for configuration." %
                (relevant_ver, ConfigFlowsDefs.FW_MIN_VERSION))


    @classmethod
    def checkOfedStarted(self, logger):
        cmd = "/etc/init.d/openibd status"
        out = ConfigFlowsCommon.commandExecute(cmd, "", False)
        for line in out.strip().split("\n"):
            if line == "":
                continue
            else:
                if line == "HCA driver is not loaded":
                    raise ConfigFlowsException("To configure RoCE OFED should be started")
                else:
                    return

    @classmethod
    def getPCIbyDevId(self, devUid):
        pciId = ""
        parts = devUid.split(":")
        if len(parts[0]) == 4 and len(parts) == 3:
            pciId = "%s:%s" % (parts[1], parts[2])
        return pciId

    @classmethod
    def getMstNameByPCIDeviceId(self, pciId):
        mst_name = ""
        cmd = "mst status -v"
        out = ConfigFlowsCommon.commandExecute(cmd, "", False)
        lines = out.splitlines()
        for line in lines:
            if line.find(pciId) != -1:
                parts = line.split()
                if len(parts) > 2:
                    mst_name = parts[1]
                break
        return mst_name

    @classmethod
    def isBlueFieldDevice(self, devUid):
        pciId = ConfigFlowsCommon.getPCIbyDevId(devUid)
        mstName = ConfigFlowsCommon.getMstNameByPCIDeviceId(pciId)
        return NeoHostCommon.isBlueFieldDevice(mstName)

#############################################################################
#########                   W I N D O W S                           #########
#############################################################################


    @classmethod
    def powerShellCommandExec(self, cmd, logger, failMsg=None, raiseExcept=True):
        pshell_cmd = 'powershell -command "%s"' % cmd
        out = ConfigFlowsCommon.commandExecute(pshell_cmd, failMsg, raiseExcept)
        logger.debug("Runnig command in powershell - %s" % cmd)
        return out


    @classmethod
    def getmlx5cmdStatData(self, param):
        cmd = 'powershell -command "mlx5cmd.exe -stat | select-string %s"' % param
        rc, out, _ = Command(cmd).execute()
        if not rc:
            lines = out.strip().splitlines()
            if len(lines) > 0:
                ret_list = lines[0].strip().split('=')
                if len(ret_list) == 2:
                    return (ret_list[1])
        return "NA"


    @classmethod
    def checkWinOFVersion(self):
        out = ConfigFlowsCommon.getmlx5cmdStatData('driver_ver')
        ver_parts = out.strip().split(".")
        if len(ver_parts) != 4:
            raise ConfigFlowsException("Cannot compare WinOF version - %s, check if driver version is newer than %s."
                  % (out, ConfigFlowsDefs.WINOF_MIN_VERSION))
        if ConfigFlowsDefs.WINOF_MIN_VERSION > out:
            raise ConfigFlowsException(
                "WinOF version %s is older than %s which needed for configuration." %
                (out, ConfigFlowsDefs.WINOF_MIN_VERSION))


    @classmethod
    def getNetworkAdapterNameByPortUid(self, port_uid, logger):
        #check port_uid format XX:XX.X
        bus_parts = port_uid.split(':')
        if len(bus_parts) != 2:
            raise ConfigFlowsException("Invalid Port UID. Expected PCI B.D.F format.")
        bus = int(bus_parts[0], 16)

        func_parts = bus_parts[1].split('.')
        if len(func_parts) != 2:
            raise ConfigFlowsException("Invalid Port UID. Expected PCI B.D.F format.")
        func = func_parts[1]

        cmd = 'Get-NetAdapterHardwareInfo | Where-Object {$_.Bus -eq %s -and $_.Function -eq %s} ' \
              '| Format-Table Name -HideTableHeaders' % (bus, func)
        out = ConfigFlowsCommon.powerShellCommandExec(cmd, logger)
        if out == "":
            raise ConfigFlowsException("Could not find Port Uid.")

        return out.strip()


    @classmethod
    def checkIfMellanoxAdapter(self, adapter_name, logger):
        cmd = 'Get-NetAdapter | Where-Object {$_.Name -eq \'%s\'} ' \
        '| Format-Table InterfaceDescription -HideTableHeaders' % adapter_name
        out = ConfigFlowsCommon.powerShellCommandExec(cmd, logger)

        lines = out.strip().splitlines()
        if len(lines)>0 and lines[0].find("Mellanox")!= -1:
            return True
        return False


    @classmethod
    def getDeviceInterfacesWindows(self, devUid, logger):
        cmd = 'Get-NetAdapter |where {$_.InterfaceDescription -like \'Mellanox *\'} | foreach { $_.Name }'
        out = ConfigFlowsCommon.powerShellCommandExec(cmd, logger)
        logger.debug(cmd)
        out.strip().split("\n")
        if out == None or out == []:
            infs_list = []
        else:
            infs_list = [item.strip() for item in ifc_list]
        return infs_list