#!/usr/bin/python
# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Ariel Gaisinsky
# @date: Jan 30, 2018


import json
import logging

from config_flows_defs import RoCEConfigType
from config_flows_defs import ConfigFlowsDefs
from config_flows_exceptions import ConfigFlowsException
from config_flows_common import ConfigFlowsCommon
from get_device_roce_state import GetDeviceRoCEState
from roce_linux_common import RoCELinuxCommon


class GetDeviceRoCEStateLinux(GetDeviceRoCEState):
    def __init__(self):
        super(GetDeviceRoCEStateLinux, self).__init__()


    def _getRoceState(self, ifc_id):
        res = {
            "configType": RoCEConfigType.NoRoCEConfiguration,
            "dscpVal": -1
        }
        try:
            ConfigFlowsCommon.checkDriverVersion(GetDeviceRoCEStateLinux.logger)
            ConfigFlowsCommon.checkOfedInstalled(GetDeviceRoCEStateLinux.logger, True)
            ConfigFlowsCommon.checkOfedStarted(GetDeviceRoCEStateLinux.logger)
            ConfigFlowsCommon.checkInterfaceExist(ifc_id, GetDeviceRoCEStateLinux.logger)
            ConfigFlowsCommon.checkIfIfcEthernet(ifc_id, GetDeviceRoCEStateLinux.logger)

            mlx_dev_name = ConfigFlowsCommon.convertInterfaceToMlxDev(ifc_id)
            ConfigFlowsCommon.checkFWVersion(mlx_dev_name, GetDeviceRoCEStateLinux.logger)

            reason = RoCELinuxCommon.checkRoCEWithoutQos(ifc_id, mlx_dev_name, GetDeviceRoCEStateLinux.logger)
            if reason is None:
                res["configType"] = RoCEConfigType.RoCEWithoutQos
            else:
                reason, tos_val = self.__checkRoCEWithQos(ifc_id, mlx_dev_name)
                if reason is None and tos_val != -1:
                    res["dscpVal"] = (tos_val - 2) / 4
                    #check priority 0
                    reason =   RoCELinuxCommon.checkPFCActivated(ifc_id,
                                                                 -1,
                                                                 GetDeviceRoCEStateLinux.logger)
                    if reason is None:
                        res["configType"] = RoCEConfigType.RoCEWithQos
                    else:
                        #check lossless
                        reason = RoCELinuxCommon.checkPFCActivated(ifc_id,
                                                                   res["dscpVal"] // 8,
                                                                   GetDeviceRoCEStateLinux.logger)
                        if reason is None:
                            res["configType"] = RoCEConfigType.LosslessRoCEWithQos

        except ConfigFlowsException as e:
            GetDeviceRoCEStateLinux.logger.error("%s" % e)
            raise e

        return res


    def __checkRoCEWithQos(self, ifc_id, mlx_dev_name):
        GetDeviceRoCEStateLinux.logger.debug("__checkRoCEWithQos")
        reason, tos_val = self.__checkRoCEWithQosSteps(ifc_id, mlx_dev_name)
        # does not check default PFC priority
        return reason, tos_val


    def __checkRoCEWithQosSteps(self, ifc_id, mlx_dev_name):
        tos_val = -1
        # step 1 - check trust dscp
        reason = RoCELinuxCommon.checkDscpAsTrust(ifc_id, "dscp", GetDeviceRoCEStateLinux.logger)
        # step 2 - get tos val
        if reason is None:
            tos_val, _ = RoCELinuxCommon.getTosVal(mlx_dev_name, GetDeviceRoCEStateLinux.logger)
        # step 3 - check cmd_roce_tos=tos_val
        if reason is None:
            reason = RoCELinuxCommon.checkRDMACMTos(mlx_dev_name, tos_val, GetDeviceRoCEStateLinux.logger)
        # step 4 - check sysctl tcp_ecn=1
        if reason is None:
            reason = RoCELinuxCommon.checkEcnenabled(1, GetDeviceRoCEStateLinux.logger)
        return reason, tos_val


