#!/usr/bin/python
# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Ariel Gaisinsky
# @date: Jan 30, 2018


from config_flows_defs import RoCEConfigType
from config_flows_exceptions import ConfigFlowsException
from config_flows_common import ConfigFlowsCommon
from get_device_roce_validation import GetDeviceRoCEValidation
from roce_linux_common import RoCELinuxCommon


class GetDeviceRoCEValidationLinux(GetDeviceRoCEValidation):
    def __init__(self):
        super(GetDeviceRoCEValidationLinux, self).__init__()

        self.__func_map = {
            RoCEConfigType.RoCEWithoutQos: self.__checkRoCEWithoutQos,
            RoCEConfigType.RoCEWithQos: self.__checkRoCEWithQos,
            RoCEConfigType.LosslessRoCEWithQos: self.__checkLosslessRoCEWithQos
        }


    def _runRoceValidation(self, ifc_id, config_type, dscp_val):
        res = {
            "msgCode": 0,
            "msgText": ""
        }
        try:
            ConfigFlowsCommon.checkDriverVersion(GetDeviceRoCEValidationLinux.logger)
            ConfigFlowsCommon.checkOfedInstalled(GetDeviceRoCEValidationLinux.logger, True)
            ConfigFlowsCommon.checkOfedStarted(GetDeviceRoCEValidationLinux.logger)
            ConfigFlowsCommon.checkInterfaceExist(ifc_id, GetDeviceRoCEValidationLinux.logger)
            ConfigFlowsCommon.checkIfIfcEthernet(ifc_id, GetDeviceRoCEValidationLinux.logger)

            mlx_dev_name = ConfigFlowsCommon.convertInterfaceToMlxDev(ifc_id)
            ConfigFlowsCommon.checkFWVersion(mlx_dev_name, GetDeviceRoCEValidationLinux.logger)

            func = self.__func_map[config_type]
            reason = func(ifc_id, mlx_dev_name, dscp_val)

            valid = True
            if reason is not None:
                valid = False
            res["roceValid"] = valid

        except ConfigFlowsException as e:
            GetDeviceRoCEValidationLinux.logger.error("%s" % e)
            raise e

        result_msg = "RoCE validation result: %s. " % res["roceValid"]
        if not valid:
            result_msg += reason
        res["msgText"] = result_msg

        return res


    def __checkRoCEWithoutQos(self, ifc_id, mlx_dev_name, dscp_val):
        return RoCELinuxCommon.checkRoCEWithoutQos(ifc_id, mlx_dev_name,
                                                 GetDeviceRoCEValidationLinux.logger)


    def __checkRoCEWithQos(self, ifc_id, mlx_dev_name, dscp_val):
        GetDeviceRoCEValidationLinux.logger.debug("__checkRoCEWithQos")

        reason = self.__checkRoCEWithQosSteps(ifc_id, mlx_dev_name, dscp_val)

        # check default PFC priority
        if reason is None:
            reason = RoCELinuxCommon.checkPFCActivated(ifc_id, -1, GetDeviceRoCEValidationLinux.logger)
        return reason


    def __checkRoCEWithQosSteps(self, ifc_id, mlx_dev_name, dscp_val):
        tos_val = dscp_val * 4 + 2

        # step 1
        reason = RoCELinuxCommon.checkDscpAsTrust(ifc_id, "dscp", GetDeviceRoCEValidationLinux.logger)
        # step 2
        if reason is None:
            reason = RoCELinuxCommon.checkTos(mlx_dev_name, tos_val, GetDeviceRoCEValidationLinux.logger)
        # step 3
        if reason is None:
            reason = RoCELinuxCommon.checkRDMACMTos(mlx_dev_name, tos_val, GetDeviceRoCEValidationLinux.logger)
        # step 4
        if reason is None:
            reason = RoCELinuxCommon.checkEcnenabled(1, GetDeviceRoCEValidationLinux.logger)
        return reason


    def __checkLosslessRoCEWithQos(self, ifc_id, mlx_dev_name, dscp_val):
        GetDeviceRoCEValidationLinux.logger.debug("__checkLosslessRoCEWithQos")
        reason = self.__checkRoCEWithQosSteps(ifc_id, mlx_dev_name, dscp_val)

        # check activated PFC on priority 3
        if reason is None:
            pri0_val = dscp_val // 8
            reason = RoCELinuxCommon.checkPFCActivated(ifc_id, pri0_val, GetDeviceRoCEValidationLinux.logger)

        return reason

