#!/usr/bin/python
# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Ariel G.N.
# @date: May 10, 2018

import json
import logging

import neohost_plugin_ifc as APIFC
from config_flows_exceptions import ConfigFlowsException
from config_flows_common import ConfigFlowsCommon
from config_flows_defs import ConfigFlowsDefs


class GetDeviceSRIOVState(APIFC.AbsNeoHostCommandIFC):
    logger = logging.getLogger("neohost." + __name__)

    def __init__(self):
        super(GetDeviceSRIOVState, self).__init__(
            cmd_name="GetDeviceSRIOVState",
            cmd_desc="get SRIOV state on device",
            cmd_type=APIFC.EnumCmdType.Cmd_Type_Set,
            cmd_scope=APIFC.EnumCmdScope.Cmd_Scope_Device,
            supp_exec_mask=APIFC.EnumCmdExecMode.Cmd_Exec_Mode_All,
            cmd_in_desc="devUid, device uid, ifcUid: interface uid ",
            cmd_out_desc="msgText: message text, msgCode: message code"
            "numVFs: number of VFS, sriovEnabled: enabled/disabled",
            extra_str="")


    def execute_command(self, json_request, exec_opt):
        ###
        req = json.loads(json_request)
        dev_uid = req["devUid"]

        if ConfigFlowsCommon.isBlueFieldDevice(dev_uid):
            raise ConfigFlowsException(
                "SRIOV functionalities are not supported for BlueField devices")

        # get device interfaces
        ifcs_list = ConfigFlowsCommon.getDeviceInterfaces(dev_uid, GetDeviceSRIOVState.logger)

        ifc_uid_list = []
        if "ifcUid" in req.keys():
            # check wanted interface to set is in device.
            if req["ifcUid"] not in ifcs_list:
                raise ConfigFlowsException(
                    "Could not find network interface %s for device %s" %
                    (req["ifcUid"], dev_uid))
            ifc_uid_list.append(req["ifcUid"])
        else:
            ifc_uid_list = ifcs_list

        # check prerequisites
        GetDeviceSRIOVState.logger.info("checking prerequisites")
        self._checkPrerequisites(ifc_uid_list)

        # get interfaces state
        res = self._getSriovState(dev_uid, ifc_uid_list)

        return ConfigFlowsDefs.CONFIG_FLOWS_STATUS_SUCCESS, json.dumps(res)


    def _getSriovState(self, dev_uid, ifc_uid):
        self.__raiseAbstarctClassException()


    def __raiseAbstarctClassException(self):
        err_msg = "GetDeviceSriovState is an abstract class, you can't create an instance."
        GetDeviceSRIOVState.logger.error(err_msg)
        raise ConfigFlowsException(err_msg)

