#!/usr/bin/python
# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Ariel Gaisinsky
# @date: May 14, 2018


from config_flows_common import ConfigFlowsCommon
from config_flows_defs import ConfigFlowsDefs


class RoCELinuxCommon:
    @classmethod
    def checkRoCEWithoutQos(self, ifc_id, mlx_dev_name, logger):
        logger.debug("checkRoCEWithoutQos")
        # step 1
        reason = RoCELinuxCommon.checkDscpAsTrust(ifc_id, "pcp", logger)
        # step 2
        if reason is None:
            reason = RoCELinuxCommon.checkTos(mlx_dev_name, -1, logger)
        # step 3
        if reason is None:
            reason = RoCELinuxCommon.checkRDMACMTos(mlx_dev_name, 0, logger)
        # step 4
        if reason is None:
            reason = RoCELinuxCommon.checkEcnenabled(1, logger)
        # step 5
        if reason is None:
            reason = RoCELinuxCommon.checkPFCActivated(ifc_id, -1, logger)
        return reason


    @classmethod
    def checkTos(self, mlx_dev, checked_val, logger):
        logger.debug("checkTos")
        tos_val, cmd = RoCELinuxCommon.getTosVal(mlx_dev, logger)
        if tos_val == checked_val:
            return None
        return ConfigFlowsDefs.ROCE_VALIDATION_FAIL_REASON % (cmd, tos_val, checked_val)


    @classmethod
    def getTosVal(self, mlx_dev, logger):
        logger.debug("getTosVal")
        cmd = "cat /sys/class/infiniband/%s/tc/1/traffic_class" % mlx_dev
        fail_msg = "Failed to check Tos. Command -%s- has failed" % cmd
        out = ConfigFlowsCommon.commandExecute(cmd, fail_msg)
        tos_val = -1
        if out != None and out != "":
            if out.startswith("Global"):
                parts = out.strip().split("=")
                if len(parts) == 2:
                    tos_val = int(parts[1].strip())
            else:
                tos_val = int(out.strip())
        return tos_val, cmd


    @classmethod
    def checkDscpAsTrust(self, ifc_id, checked_val, logger):
        logger.debug("checkDscpAsTrust")
        cmd = "mlnx_qos -i %s" % ifc_id
        fail_msg = "Failed to check DSCP as trust. Command -%s- has failed." % cmd
        out = ConfigFlowsCommon.commandExecute(cmd, fail_msg)
        for line in out.splitlines():
            if line.find("Priority trust") != -1:
                line_parts = line.split()
                cur_val = line_parts[len(line_parts) - 1].strip()
                if cur_val == checked_val:
                    return None
        return ConfigFlowsDefs.ROCE_VALIDATION_FAIL_REASON % (cmd, cur_val, checked_val)


    @classmethod
    def checkRDMACMTos(self, mlx_dev, checked_val, logger):
        logger.debug("checkRDMACMTos")
        cmd = "cma_roce_tos -d %s" % mlx_dev
        fail_msg = "Failed to check Tos. Command -%s- has failed" % cmd
        out = ConfigFlowsCommon.commandExecute(cmd, fail_msg)
        if out.strip() == str(checked_val):
            return None
        return ConfigFlowsDefs.ROCE_VALIDATION_FAIL_REASON % (cmd, out.strip(), checked_val)


    @classmethod
    def checkEcnenabled(self, checked_val, logger):
        logger.debug("checkEcnenabled")
        cmd = "sysctl -n net.ipv4.tcp_ecn"
        fail_msg = "Failed to check Enc enabled. Command -%s- has failed" % cmd
        out = ConfigFlowsCommon.commandExecute(cmd, fail_msg)
        if out.strip() == str(checked_val):
            return None
        return ConfigFlowsDefs.ROCE_VALIDATION_FAIL_REASON % (cmd, out.strip(), checked_val)


    @classmethod
    def checkPFCActivated(self, ifc_id, checked_val, logger):
        logger.debug("activatePFC")
        cmd = "mlnx_qos -i %s" % ifc_id
        fail_msg = "Failed to check activating PFC on priority 3. Command -%s- has failed." % cmd
        out = ConfigFlowsCommon.commandExecute(cmd, fail_msg)
        mlnx_qos_list = out.split('\n')

        if checked_val == -1:
            fail_reason = "-%s- command failed. expecting 0,0,0,0,0,0,0,0" % cmd
        else:
            fail_reason = "-%s- command failed. only bit in index(%d) should " \
                          "be set to " % (cmd, checked_val)

        # string =>  enabled     0   0   0   1   0   0   0   0
        for line in mlnx_qos_list:
            line = line.strip()
            if line.startswith('enabled'):
                # split line and ignore spaces
                en_parts = [s.strip() for s in line.split('  ') if s]
                index = 0
                en_parts.remove('enabled')
                if len(en_parts) > 1:
                    for part in en_parts:
                        if index == checked_val:
                            if part != "1":
                                return fail_reason
                        if index >= 0 and index != checked_val:
                            if part != "0":
                                return fail_reason
                        index += 1
                    return None
                else:
                    return fail_reason
        return fail_reason