#!/usr/bin/python
# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Ariel Gaisinsky
# @date: Jan 30, 2018

import json
import logging

from config_flows_defs import RoCEConfigType
from config_flows_exceptions import ConfigFlowsException
from config_flows_common import ConfigFlowsCommon
from set_device_roce import SetDeviceRoCE


class SetDeviceRoCELinux(SetDeviceRoCE):
    def __init__(self):
        super(SetDeviceRoCELinux, self).__init__()

        self.__func_map = {
            RoCEConfigType.RoCEWithoutQos: self.__configureRoCEWithoutQos,
            RoCEConfigType.RoCEWithQos: self.__configureRoCEWithQos,
            RoCEConfigType.LosslessRoCEWithQos:
                self.__configureLosslessRoCEWithQos
        }

        self.__prio_map = {
            0: "1,0,0,0,0,0,0,0",
            1: "0,1,0,0,0,0,0,0",
            2: "0,0,1,0,0,0,0,0",
            3: "0,0,0,1,0,0,0,0",
            4: "0,0,0,0,1,0,0,0",
            5: "0,0,0,0,0,1,0,0",
            6: "0,0,0,0,0,0,1,0",
            7: "0,0,0,0,0,0,0,1",
        }


    def _runConfigureCommand(self, config_type, ifc_id, dscp_val, force):
        try:
            #checks
            self.__checkDscpLimitations(dscp_val, force)

            # cleanup before configure
            SetDeviceRoCELinux.logger.debug("Runnig clean command...")
            self._cleanConfiguration(ifc_id)

            # run configuration command
            func = self.__func_map[config_type]
            mlx_dev_name = ConfigFlowsCommon.convertInterfaceToMlxDev(ifc_id)
            func(ifc_id, mlx_dev_name, dscp_val)
            SetDeviceRoCELinux.logger.debug("Finished RoCe configuration.")

        except ConfigFlowsException as e:
            SetDeviceRoCELinux.logger.error("%s" % e)
            raise e


    def __configureRoCEWithoutQos(self, ifc_id, mlx_dev_name, dscp_val):
        SetDeviceRoCELinux.logger.debug("Lossy fabric Already preconfigured. Nothing to do. using clean command.")
        # do nothing - only cleanup


    def __configureRoCEWithQos(self, ifc_id, mlx_dev_name, dscp_val):
        SetDeviceRoCELinux.logger.debug("__configureRoCEWithQos")
        self.__preValidateRoCEConfig()

        # step 1
        self.__setDscpAsTrust(ifc_id)

        # ste 2
        self.__setTos(mlx_dev_name, dscp_val)

        # step 3
        self.__setRDMACMTos(mlx_dev_name, dscp_val)

        # step 4
        self.__enableECN()


    def __preValidateRoCEConfig(self):
        SetDeviceRoCELinux.logger.debug("__preValidateRoCEConfig")
        ConfigFlowsCommon.preValidateCommand("mlnx_qos", SetDeviceRoCELinux.logger)
        ConfigFlowsCommon.preValidateCommand("sysctl", SetDeviceRoCELinux.logger)
        ConfigFlowsCommon.preValidateCommand("cma_roce_tos", SetDeviceRoCELinux.logger)


    def __configureLosslessRoCEWithQos(self, ifc_id, mlx_dev_name, dscp_val):
        SetDeviceRoCELinux.logger.debug("__configureLosslessRoCEWithQos")
        self.__configureRoCEWithQos(ifc_id, mlx_dev_name, dscp_val)

        # step 4 - Activate PFC on priority 3
        self.__activatePFC(ifc_id, dscp_val)


    def __setDscpAsTrust(self, ifc_id):
        SetDeviceRoCELinux.logger.debug("__setDscpAsTrust")
        cmd = "mlnx_qos -i %s --trust dscp" % ifc_id
        fail_msg = "Failed to set DSCP as trust. Command -%s- has failed." % cmd
        ConfigFlowsCommon.commandExecute(cmd, fail_msg)


    def __setTos(self, mlx_dev, dscp_val):
        SetDeviceRoCELinux.logger.debug("__setTos")
        tos_val = dscp_val * 4 + 2
        cmd = "echo %d > /sys/class/infiniband/%s/tc/1/traffic_class" % (tos_val, mlx_dev)
        fail_msg = "Failed to set Tos. Command -%s- has failed" % cmd
        ConfigFlowsCommon.commandExecute(cmd, fail_msg)


    def __setRDMACMTos(self, mlx_dev, dscp_val):
        SetDeviceRoCELinux.logger.debug("__setRDMACMTos")
        tos_val = dscp_val * 4 + 2
        cmd = "cma_roce_tos -d %s -t %d" % (mlx_dev, tos_val)
        fail_msg = "Failed to set RDMA-CM Tos. Command -%s- has failed" % cmd
        ConfigFlowsCommon.commandExecute(cmd, fail_msg)


    def __enableECN(self):
        SetDeviceRoCELinux.logger.debug("__enableECN")
        cmd = "sysctl -w net.ipv4.tcp_ecn=1"
        fail_msg = "Failed to enable ECN"
        ConfigFlowsCommon.commandExecute(cmd, fail_msg)
        SetDeviceRoCELinux.logger.debug("enabled ECN for TCP traffic. ")


    def __activatePFC(self, ifc_id, dscp_val):
        SetDeviceRoCELinux.logger.debug("__activatePFC")
        pri0_val = dscp_val // 8
        cmd = "mlnx_qos -i %s --pfc %s" % (ifc_id, self.__prio_map[pri0_val])
        fail_msg = "Failed to activate PFC on priority 3. Command -%s- has failed." % cmd
        ConfigFlowsCommon.commandExecute(cmd, fail_msg)
        SetDeviceRoCELinux.logger.debug("Activated PFC on priority 3.")


    def __checkDscpLimitations(self, dscp_val, force):
        # if force flag is not on check limitations
        if not force:
            prio = dscp_val // 8
            if prio == 0:
                raise ConfigFlowsException("Cannot set RoCE configuration. Supplied DSCP value is reserved"
                                           " for non RDMA traffic. If you still want to use this value please "
                                           "use the force parameter.")
            if prio == 0:
                raise ConfigFlowsException("Cannot set RoCE configuration. Supplied DSCP value is reserved"
                                           " for CNP (Congestion Notification Packets). If you still want to use "
                                           "this value please use the force parameter.")
