#!/usr/bin/python
# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Samer Deeb
# @date: Dec 12, 2016

import json
import logging

import neohost_plugin_ifc as APIFC
from config_flows_exceptions import ConfigFlowsException
from config_flows_common import ConfigFlowsCommon
from config_flows_defs import ConfigFlowsDefs
from request_factory import RequestFactory
from msg_mgr import MsgMgr, MsgConstants


class SetDeviceSRIOV(APIFC.AbsNeoHostCommandIFC):
    logger = logging.getLogger("neohost." + __name__)

    def __init__(self):
        super(SetDeviceSRIOV, self).__init__(
            cmd_name="SetDeviceSRIOV",
            cmd_desc="set/unset SRIOV on device",
            cmd_type=APIFC.EnumCmdType.Cmd_Type_Set,
            cmd_scope=APIFC.EnumCmdScope.Cmd_Scope_Device,
            supp_exec_mask=APIFC.EnumCmdExecMode.Cmd_Exec_Mode_All,
            cmd_in_desc="devUid, device uid, numVFsConfig: VFs number "
            "per interface, updateFw: flag for updating FW",
            cmd_out_desc="msgText: message text, msgCode: message code",
            extra_str="")


    def execute_command(self, json_request, exec_opt):
        ###
        req = json.loads(json_request)
        dev_uid = req["devUid"]
        num_vfs_config = req["numVFsConfig"]
        updateFw = req.get("updateFw", False)

        ConfigFlowsCommon.checkDriverVersion(SetDeviceSRIOV.logger)

        if ConfigFlowsCommon.isBlueFieldDevice(dev_uid):
            raise ConfigFlowsException(
                "SRIOV functionalities are not supported for BlueField devices")

        # get pairs map {ifcUid, VFS_NUM}
        vf_config_mapping = {}
        for vf_config in num_vfs_config:
            ifc_uid = vf_config["ifcUid"]
            if ifc_uid in vf_config_mapping:
                raise ConfigFlowsException(
                    "Duplicate entries for interface %s" % ifc_uid)
            vf_config_mapping[ifc_uid] = vf_config["numVFs"]

        # get device interfaces
        ifc_list = ConfigFlowsCommon.getDeviceInterfaces(dev_uid, SetDeviceSRIOV.logger)

        # check wanted interface to set is in device.
        # calc the max num_vfs needed
        max_vfs = 0
        set_ifc_list = []
        for ifc_uid, num_vfs in vf_config_mapping.iteritems():
            if ifc_uid not in ifc_list:
                raise ConfigFlowsException(
                    "Could not find network interface %s for device %s" %
                    (ifc_uid, dev_uid))
            else:
                set_ifc_list.append(ifc_uid)
            if num_vfs > max_vfs:
                max_vfs = num_vfs
        enable_sriov = 0 if max_vfs == 0 else 1

        # check prerequisites
        SetDeviceSRIOV.logger.info("checking prerequisites")
        self._checkPrerequisites(set_ifc_list)

        if enable_sriov:
            SetDeviceSRIOV.logger.info("setting SRIOV to enable")
            res = self._enableSriov(dev_uid, vf_config_mapping, max_vfs, updateFw)
        else:
            SetDeviceSRIOV.logger.info("setting SRIOV to disable")
            res = self._disableSriov(dev_uid, vf_config_mapping, updateFw)

        return ConfigFlowsDefs.CONFIG_FLOWS_STATUS_SUCCESS, json.dumps(res)


    def _enableSriov(self, dev_uid, vf_config_mapping, max_vfs, updateFw):
        self.__raiseAbstarctClassException()


    def _disableSriov(self, dev_uid, vf_config_mapping, updateFw):
        self.__raiseAbstarctClassException()


    def __raiseAbstarctClassException(self):
        err_msg = "SetDeviceSriov is an abstract class, you can't create an instance."
        SetDeviceSRIOV.logger.error(err_msg)
        raise ConfigFlowsException(err_msg)


    def _configureDevice(self, devUid, num_vfs, enable_sriov):
        params_list = []

        if enable_sriov != None:
            item = {"paramName": "SRIOV_EN",
                    "paramVal": str(enable_sriov)}
            params_list.append(item)

        # if num_vfs not none - add it
        if num_vfs is not None:
            item = {"paramName": "NUM_OF_VFS",
                    "paramVal": str(num_vfs)}
            params_list.append(item)

        config_params = {
            "devUid": devUid,
            "paramsList": params_list
        }

        SetDeviceSRIOV.logger.debug("config FW with params: %s" % config_params)

        sub_req_obj = RequestFactory.createRequest(
            1, "mftConfig", "SetDeviceNVConfig", config_params)
        _, rc, response = MsgMgr().handle(
            MsgConstants.NEOHOST_REQUEST, sub_req_obj)
        if rc != 0:
            raise ConfigFlowsException(
                "Failed to configure device %s: %s" %
                (devUid, response["message"]))


  
