#!/usr/bin/python
# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Samer Deeb
# @date: Dec 12, 2016

import logging

from config_flows_exceptions import ConfigFlowsException
from set_device_sriov import SetDeviceSRIOV
from config_flows_common import ConfigFlowsCommon
from sriov_common import SRIOVCommon


class SetDeviceSRIOVWindows(SetDeviceSRIOV):
    def __init__(self):
        super(SetDeviceSRIOVWindows, self).__init__()


    def _checkPrerequisites(self, ifc_list):
        exp_msg = "SRIOV is not supported. Please check BIOS configured correctly and " \
                  "Hypervisor Operating System installed as required in WinOF2 user manual." \
                  "For more information please check the logger"

        cmd = '(Get-VmHost)'
        out = ConfigFlowsCommon.powerShellCommandExec(cmd, SetDeviceSRIOVWindows.logger, exp_msg, False)
        if out != "":
            apnd = "Get-VmHost is unknown. "
            SetDeviceSRIOVWindows.logger.error(apnd + exp_msg)
            raise ConfigFlowsException(apnd + exp_msg)

        cmd = '(Get-VmHost).IovSupport'
        out = ConfigFlowsCommon.powerShellCommandExec(cmd, SetDeviceSRIOVWindows.logger, exp_msg, False)
        if out.strip() != "True":
            apnd = "(Get-VmHost).IovSupport command returned %s" % out
            SetDeviceSRIOVWindows.logger.error(exp_msg + apnd)
            raise ConfigFlowsException(exp_msg + apnd)

        cmd = '(Get-VmHost).IovSupportReasons'
        out = ConfigFlowsCommon.powerShellCommandExec(cmd, SetDeviceSRIOVWindows.logger, exp_msg, False)
        if out.strip() != "OK":
            apnd = "(Get-VmHost).IovSupportReasons command returned %s" % out
            SetDeviceSRIOVWindows.logger.error(exp_msg + apnd)
            raise ConfigFlowsException(exp_msg + apnd)


    def _disableSriov(self, dev_uid, vfs_config_map, updateFw):
        res = {
            "msgCode": 0,
            "msgText": "Success"
        }

        # set num of vafs to 0 in the driver files and sriov disable
        SetDeviceSRIOVWindows.logger.info("setting OS params to SRIOV enable")
        self.__enableOsSriov(vfs_config_map, False)
        SetDeviceSRIOVWindows.logger.info("changing OS NUmVFs")
        self.__changeOsNumOfVFs(vfs_config_map, True)

        # only if user used updateFw attr with True
        if updateFw:
            curr_num_vfs, curr_enable_sriov = SRIOVCommon.getCurFwSriovConfig(dev_uid)

            if curr_enable_sriov :
                # in case of disable - we DON'T change the NUM_OF_VFS number in the FW to 0
                self._configureDevice(dev_uid, None, "False")
                res["msgCode"] = 2
                res["msgText"] = "Operation succeeded. However, you need to reboot " \
                               "in order to complete the changes in the device FW, " \
                               "and then run the operation again"
        return res


    def _enableSriov(self, dev_uid, vfs_config_map, max_vfs, updateFw):
        curr_num_vfs, curr_enable_sriov = SRIOVCommon.getCurFwSriovConfig(dev_uid)
        SetDeviceSRIOVWindows.logger.debug("enable sriov - current status: cur_num=%s, cur_enable=%s" %
                    (curr_num_vfs, curr_enable_sriov))
        # if user asked more num of vfs them current and no used update_fw flag with true
        if curr_num_vfs < max_vfs and not updateFw:
            res = {
                "msgCode": 2,
                "msgText": "Failed to configure SRIOV: Num_oF_VFS requested is grater than the current if the FW. "
                "To update it please use the updateFw flag with True."
            }
            return res

        if not curr_enable_sriov or curr_num_vfs < max_vfs:
            # no meaning for updateFw - will enable Sriov in FW anyway
            self._configureDevice(dev_uid, max_vfs, "True")
            res = {
                "msgCode": 1,
                "msgText": "Further operation needed, you have to reboot "
                           "for the changes to take effect, "
                           "after that please rerun the command "
                           "again to configuring the interface"
            }
            return res

        # enable sriov in os
        SetDeviceSRIOVWindows.logger.info("enabling SRIOV in OS")
        self.__enableOsSriov(vfs_config_map, True)

        # change num of VFS in OS
        SetDeviceSRIOVWindows.logger.info("changing NumVFs in OS")
        self.__changeOsNumOfVFs(vfs_config_map)

        # check sriov supported
        SetDeviceSRIOVWindows.logger.info("checking sriov supported in OS")
        self.__checkAdapterSriov(vfs_config_map)

        res = {
            "msgCode": 0,
            "msgText": "Success"
        }
        return res


    def __enableOsSriov(self, vfs_config_map, enable):
        for ifc_name in vfs_config_map.keys():
            set_val = "True" if enable else "False"
            cmd = 'Set-NetAdapterSriov -Name \'%s\' -Enabled $%s' % (ifc_name, set_val)
            SetDeviceSRIOVWindows.logger.debug(cmd)
            out = ConfigFlowsCommon.powerShellCommandExec(cmd, SetDeviceSRIOVWindows.logger)


    def __changeOsNumOfVFs(self, vfs_config_map, reset=False):
        for ifc_name, req_vfs in vfs_config_map.items():
            vfs = 0 if reset else req_vfs
            cmd = 'Set-NetAdapterSriov -Name \'%s\' -NumVFs %s' % (ifc_name, vfs)
            SetDeviceSRIOVWindows.logger.debug(cmd)
            ConfigFlowsCommon.powerShellCommandExec(cmd, SetDeviceSRIOVWindows.logger)


    def __checkAdapterSriov(self, vfs_config_map):
        for ifc_name in vfs_config_map.keys():
            cmd = 'Get-NetAdapterSriov \'%s\' | foreach {$_.SriovSupport}' % ifc_name
            out = ConfigFlowsCommon.powerShellCommandExec(cmd, SetDeviceSRIOVWindows.logger)
            SetDeviceSRIOVWindows.logger.debug(cmd)
            if out.strip() == "Supported":
                break
            else:
                raise ConfigFlowsException("SRIOV is not supported for %s in Get-NetAdapterSriov. Please check "
                                           "SRIOV_EN and NUM_VFS in FW." % ifc_name)
