#!/usr/bin/python
# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# Python Imports
import json
import sys
import re
from cmd_exec import Command
import json_entry_creator
# Plugin Imports
import mft_core_exceptions
from mft_core_defs import MftCoreDefs
import os
from common_help_funcs import CommonHelpFuncs
from neohost_common import NeoHostCommon
from get_port_info import GetPortInfoCommand


def _atoi(numstr):
    if type(numstr) is int:
        return numstr
    num = 0
    if type(numstr) not in (str, unicode):
        return num
    numstr = str(numstr)
    for c in numstr:
        if c in (' ', '\t'):
            continue
        if '0' <= c <= '9':
            num = num * 10 + ord(c) - ord('0')
        else:
            break
    return num

def _atof(numstr):
    if type(numstr) is float:
        return numstr
    return float(numstr)

class GetPortInfoCommandLinux(GetPortInfoCommand):
    PORT_ATTR_MAPPING_LINUX = {
        "link layer": ("linkLayer", "Link Layer",
                       "Current port Link Layer configuration.", None, 0),
        "rate": ("speed", "Speed", "Port Speed in Gb/s.", _atof, 1),
        "state": ("state", "Link State",
                  "Port state (affected by the active subnet manager).",
                  None, 2),
        "physical state": ("phyState", "Physical State",
                           "Port physical state.", None, 3),
        "base lid": ("baseLid", "Base LID",
                     "The local identifier "
                     "(assigned by the active Subnet Manager).",
                     _atoi, 4),
        "lmc": ("lmc", "LMC", "LID Mask Control", _atoi, 5),
        "sm lid": ("smLid", "SM LID",
                   "The Subnet Manager's local identifier.", _atoi, 6),
        "capability mask": ("capabilityMask", "Capability Mask",
                            "Port Capabilities mask", None, -1),
        "port guid": ("portGuid", "Port GUID", "GUID assigned to the port.",
                      None, 8),
    }

    Name = {
        "name": ("Name", "Device Name")
    }


    def __init__(self):
        super(GetPortInfoCommandLinux, self).__init__()


    def _getPortInfo(self, portUid, entry_creator):
        result_dict = dict()
        local_idx = 0

        try:
            dev_path = "/sys/bus/pci/devices/%s/infiniband" % portUid
            dirs = os.listdir(dev_path)
        except OSError:
            dirs = []

        if not dirs:
            CommonHelpFuncs.printToLogAndRaiseMftCoreException(
                "Failed to get port information for: %s, path not found or empty: %s" % (portUid, dev_path),
                GetPortInfoCommandLinux.logger)

        mlx_port_id = dirs[0]
        port_info_dict = self.__get_ibstat_dict(mlx_port_id)
        lane_rate_label_parser = re.compile(r"(\d+(\.\d+)?)((\s+)?.+)?")
        parse_obj = None
        for port_info_attr, (attr_name, attr_disp, attr_desc, cast, index) in\
                self.PORT_ATTR_MAPPING_LINUX.iteritems():
            local_idx = index
            if port_info_attr in port_info_dict:
                val = port_info_dict[port_info_attr].strip()
                if cast:
                    parse_obj = lane_rate_label_parser.match(val)
                    if parse_obj:
                        val = cast(parse_obj.group(1))
                        if port_info_attr == "rate":
                            lane_rate_label = parse_obj.group(3)
                            val = "%sGb/s%s" % (val, lane_rate_label \
                                                if lane_rate_label is not None \
                                                else "")
            else:
                val = "N/A"
            result_dict[attr_name] = entry_creator.create_entry(
                attr_disp, val, attr_desc, local_idx)

        attr_name = "name"
        attr_val = mlx_port_id
        attr_disp, attr_desc = self.Name.get(attr_name)
        result_dict[attr_name] = entry_creator.create_entry(
            attr_disp, attr_val, attr_desc, local_idx)
        local_idx += 1

        return result_dict


    def __get_ibstat_dict(self, mlx5_x_str):
        cmd = "ibstat %s" % mlx5_x_str
        rc, out, _ = Command(cmd).execute()
        if rc != 0:
            CommonHelpFuncs.printToLogAndRaiseMftCoreException(
                        "Failed to get port information for: %s" % mlx5_x_str,
                    GetPortInfoCommandLinux.logger)

        port_info_dict = dict()
        for line in out.splitlines():
            if "Port 2" in line:
                # this shouldn't happen ! unless FPP is not active
                CommonHelpFuncs.printToLogAndRaiseMftCoreException(
                        "Unsupported port: %s" % mlx5_x_str,
                        GetPortInfoCommandLinux.logger)

            line = line.strip().split(":", 1)
            if len(line) < 2:
                continue
            port_info_dict[line[0].lower().strip()] = line[1]
        return port_info_dict

