#!/usr/bin/python
# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


try:
    # Python Imports
    import sys
    import platform
    import multiprocessing
    import json
    import socket
    import os
    # NeoHost IFC Imports
    import neohost_plugin_ifc as APIFC
    import json_entry_creator
    # Plugin Imports
    from mft_core_defs import MftCoreDefs
    from cmd_exec import Command
    import mft_core_exceptions
    import logging
    from neohost_common import NeoHostCommon
    # Conditional Imports
    if  NeoHostCommon.isWindowsOs():
        import psutil
except Exception as e:
    print("-E- could not import : %s" % str(e))
    sys.exit(1)

logger = logging.getLogger("neohost." + __name__)

class GetSystemServerInfoCommand(APIFC.AbsNeoHostCommandIFC):
    def __init__(self):
        super(GetSystemServerInfoCommand, self).__init__(
            cmd_name="GetSystemServerInfo",
            cmd_desc="get various system information",
            cmd_type=APIFC.EnumCmdType.Cmd_Type_Get,
            cmd_scope=APIFC.EnumCmdScope.Cmd_Scope_System,
            supp_exec_mask=APIFC.EnumCmdExecMode.Cmd_Exec_Mode_All,
            cmd_in_desc="null or object contains attrMask",
            cmd_out_desc="various system information",
            extra_str="")
        self.SERVER_ATTRS = [
            ("hostName", "Host Name", socket.gethostname,
                "Name of the host.", None),
            ("os", "Operating System",
                self.__get_os_dist,
                "operating system. distribution and version", None),
            ("kernel", "Kernel", self.__get_kernel_version,
                "Installed kernel version.", None),
            ("cpuArch", "CPU Architecture", platform.machine,
                "CPU architecture.", None),
            ("cpuCores", "CPU Cores", multiprocessing.cpu_count,
                "Number of CPU cores.", int),
            ("totalMem", "Total Memory (GB)", self.__get_total_mem,
                "Total available memory in the system.", None),
            ("ofedVer", "Driver", self.__getOfedVersion,
             "OFED driver version.", None)
        ]

    def __get_os_dist(self):
        if NeoHostCommon.isWindowsOs():
            cmd = 'powershell -command "(Get-WmiObject -class Win32_OperatingSystem).Caption"'
            rc, out, _ = Command(cmd).execute()
            if not rc:
                return out
            else:
                return "%s %s %s" % \
                     (platform.system(), platform.release(), platform.version())
        elif NeoHostCommon.isFreebsdOs():
            distname = platform.uname()[0]
            version = platform.uname()[2]
            return "%s %s" % (distname, version)
        else:
            distname, version, os_id = platform.linux_distribution()
            if os_id:
                return "%s %s (%s)" % (distname.strip(), version, os_id)
            return "%s %s" % (distname, version)

    def __get_kernel_version(self):
        return platform.uname()[2]

    def __get_total_mem(self):
        if not NeoHostCommon.isWindowsOs():
            mem_in_bytes = os.sysconf("SC_PAGE_SIZE") * os.sysconf("SC_PHYS_PAGES")
            mem_in_gigabytes = float(mem_in_bytes) / (1024 ** 3)
        else:
            mem_in_gigabytes = psutil.virtual_memory().total / (1024.0 ** 3)
        return "%.2f GB" % mem_in_gigabytes

    def execute_command(self, json_request, exec_opt):
        logger.info("Executing GetSystemServerInfo command..")
        req = json.loads(json_request)
        result_dict = dict()
        localIdx = 0
        entry_creator = json_entry_creator.InfoEntryCreator()
        if req:
            attrMask = req["attrMask"]
            entry_creator.set_attr_mask(attrMask)

        for (attr, disp, method, desc, cast) in self.SERVER_ATTRS:
            val = method()
            if cast:
                val = cast(val)
            index = localIdx
            if NeoHostCommon.isWindowsOs() and attr == "kernel":
                index = -1
            else:
                localIdx += 1
            result_dict[attr] = entry_creator.create_entry(
                disp, val, desc, index)

        final_result = dict()
        final_result["serverInfo"] = entry_creator.create_entry(
            "Server Information", result_dict,
            "Information about the server.", 0)
        logger.info("finished executing GetSystemServerInfo command.")
        return MftCoreDefs.MFT_CORE_STATUS_SUCCESS, json.dumps(final_result)

    def __getOfedVersion(self):
        if not NeoHostCommon.isWindowsOs():
            ret= "N/A"
            cmd = "ofed_info | grep MLNX_OFED_LINUX"
            out = ""
            rc, out, _ = Command(cmd).execute()
            if rc:
                logger.info("OFED is not installed.")
            else:
                strList = out.split(" ")
                ret = strList[0]
            return ret
        else:
            win_ver_str = self.__getWinofVer()
            return "MLNX_WinOF2-" + win_ver_str


    def __getWinofVer(self):
        cmd = 'mlx5cmd.exe -stat'
        rc, out, _ = Command(cmd).execute()
        if not rc:
            lines = out.strip().splitlines()
            for line in lines:
                if line.find('driver_ver') != -1:
                    deriver_parts = line.split('=')
                    if len(deriver_parts) != 2:
                        return "N/A"
                    else:
                        return deriver_parts[1]
                    break
        return "N/A"
