#!/usr/bin/python
# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


import sys
import socket
import optparse
import subprocess

###########################################################
#            CONST GLOBAL VARIABLES

# Linux Server Related Commands
DMIDECODE_S_BIOS_VERSION = "dmidecode -s bios-version"
DMIDECODE_S_BIOS_RELEASE_DATE = "dmidecode -s bios-release-date"
PROC_CPUINFO = "cat /proc/cpuinfo"
GRUB_x86_64_1 = "cat /boot/grub/grub.conf"
GRUB_x86_64_2 = "cat /boot/grub/grub.cfg"
GRUB_PPC_1 = "cat /boot/grub2/grub.cfg"
GRUB_PPC_2 = "cat /boot/grub2/grub.conf"
DMIDECODE_S_SYSTEM_PRODUCT_NAME = "dmidecode -s system-product-name"
IFCONFIG = "ifconfig"
IP_A_S = "ip a s"
DMIDECODE_S_SYSTEM_MANUFACTURER = "dmidecode -s system-manufacturer"
DMIDECODE_S_SYSTEM_SERIAL_NUMBER = "dmidecode -s system-serial-number"

# ESXi VM Related Commands
ESXI_BIOS_INFO = "vim-cmd hostsvc/hosthardware | grep biosInfo -A 2"
ESXI_CPU_INFO = "vim-cmd hostsvc/hosthardware | grep cpuPkg -A 6"
ESXI_SYSTEM_PRODUCT_NAME = "esxcfg-info | grep " + '"' + "Product Name" + \
    '"' + " -m 1"
ESXI_VENDOR_NAME = "esxcfg-info | grep " + '"' + "Vendor Name" + '"' + " -m 1"
ESXI_SERIAL_NUMBER = "esxcfg-info | grep " + '"' + "Serial Number" + '"' + \
    " -m 1"

# Windows Server Related Commands
WMIC_BIOS_GET_SMBIOSBIOSVERSION = "wmic bios get smbiosbiosversion"
WMIC_BIOS_GET_RELEASEDATE = "wmic bios get ReleaseDate"
WMIC_CPU_GET_NAME = "wmic cpu get name"
BCDEDIT = "bcdedit"  # Shows operating systems list in windows
WMIC_COMPUTERSYSTEM_GET_MODEL = "wmic computersystem get model"
# Equivalent to "wmic csproduct get name"

IPCONFIG = "ipconfig"
WMIC_COMPUTERSYSTEM_GET_MANUFACTURER = "wmic computersystem get manufacturer"
WMIC_BIOS_GET_SERIALNUMBER = "wmic bios get SerialNumber"

MAX_OS = 3
MAX_HCA = 3
MAX_PCIE = 3

PRESENT = "PRESENT"
NOT_QUERIED = "Not Queried"
NOT_PRESENT = "Not Present"
NOT_AVAILABLE = "N/A"

OEM_collection = ["NON-OEM", "HP", "HEWLETT", "HEWLETT-PACKARD", "IBM",
                  "DELL", "SGI", "FUJITSU", "EMC", "NETAPP", "TERADATA",
                  "MICROSOFT", "FACEBOOK", "QUANTA", "QUANTA-COMPUTER",
                  "HUAWEI", "BULL"]

###########################################################
#        GLOBAL VARIABLES

version = "1.0.1"
g_verbosity = False
g_operating_system = None


###########################################################
#        Linux Info


def exec_command(cmd_str):
    p = subprocess.Popen(cmd_str,
                         stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE,
                         stderr=subprocess.PIPE,
                         universal_newlines=False,
                         shell=True)
    stdout, stderr = p.communicate()
    stat = p.wait()
    return stat, stdout, stderr


class Bios:
    def __init__(self):
        if g_verbosity:
            print('\tQuery Bios - Start')
        self.version = ""
        self.release_date = ""
        self.bios = ""

        stat, stdout, _ = exec_command(DMIDECODE_S_BIOS_VERSION)
        if stat == 0:
            self.version = stdout.splitlines()[-1]
        stat, stdout, _ = exec_command(DMIDECODE_S_BIOS_RELEASE_DATE)
        if stat == 0:
            self.release_date = stdout.splitlines()[-1]
        if self.version == "" and self.release_date == "":
            self.bios = NOT_AVAILABLE
        else:
            self.bios = self.version + " | " + self.release_date
        if g_verbosity:
            print('\tQuery Bios - End')

# ----------------------------------------------------------


class CPU:
    def __init__(self):
        if g_verbosity:
            print('\tQuery CPU - Start')
            print('\t\tQuery PCIE Generation Capability - Start')
        self.cpu = NOT_AVAILABLE
        self.pcie_gen_cap = NOT_AVAILABLE

        stat, stdout, _ = exec_command(PROC_CPUINFO)
        if stat == 0:
            cpu = stdout
            if "model name" in cpu:
                cpu = cpu.split("model name")[1].splitlines()[0]
                if ": " in cpu:
                    self.cpu = cpu.split(": ")[1]
                    if "e5" in self.cpu.lower():
                        self.pcie_gen_cap = "3"
                    else:
                        self.pcie_gen_cap = "2"
            elif "cpu architecture" in cpu.lower():
                pcie_gen_cap = cpu.lower().split(
                    "cpu architecture")[1].splitlines()[0]
                if ": " in pcie_gen_cap:
                    pcie_gen_cap = pcie_gen_cap.split(": ")[1]
                    if "aarch64" in pcie_gen_cap:
                        self.pcie_gen_cap = "3"
                    else:
                        self.pcie_gen_cap = "2"
        if g_verbosity:
            print('\t\tQuery PCIE Generation Capability - End')
            print('\tQuery CPU - End')

# ----------------------------------------------------------


class BOOT_GRUB_CONF:
    def __init__(self):
        if g_verbosity:
            print('\t\tQuery Boot Grub Conf - Start')
        self.os = [NOT_AVAILABLE, NOT_AVAILABLE, NOT_AVAILABLE]
        self.kernel = [NOT_AVAILABLE, NOT_AVAILABLE, NOT_AVAILABLE]

        count = 0

        stat, stdout, _ = exec_command(GRUB_x86_64_1)
        if stat == 0:
            count = 1
        else:
            stat, stdout, _ = exec_command(GRUB_x86_64_2)
            if stat == 0:
                count = 2
            else:
                stat, stdout, _ = exec_command(GRUB_PPC_1)
                if stat == 0:
                    count = 3
                else:
                    stat, stdout, _ = exec_command(GRUB_PPC_2)
                    if stat == 0:
                        count = 4
                    else:
                        if g_verbosity:
                            print('\t\tQuery Boot Grub Conf - End')
                        return
        i = 0
        if count == 1 or count == 4:
            grub = stdout
            grub = grub.splitlines()
            i = 0
            xen_gz = False
            for line in grub:
                if (i > 2):
                    if g_verbosity:
                        print('\t\tQuery Boot Grub Conf - End')
                    return
                line = line.strip()
                if line.startswith("title "):
                    self.os[i] = line.split("x86")[0].split("title ")[1]
                elif line.startswith("kernel ") or \
                        line.startswith("\tkernel "):
                    if "xen.gz" in line:
                        xen_gz = True
                    else:
                        if len(line.split()) > 2:
                            line = line.split()[1]
                        line = line.split("-")
                        while len(line) > 1 and line[0][0].isdigit() is False:
                            line.pop(0)
                        self.kernel[i] = '-'.join(line)
                        i += 1
                elif line.startswith("module") and xen_gz:
                    xen_gz = False
                    if len(line.split()) > 2:
                        line = line.split()[1]
                    line = line.split("-")
                    while (len(line) > 1 and line[0][0].isdigit() is False):
                        line.pop(0)
                    self.kernel[i] = '-'.join(line)
                    i += 1
        elif count == 2 or count == 3:
            grub = stdout
            i = 0
            grub = grub.split("menuentry '")
            grub.pop(0)
            for distro in grub:
                if (i > 2):
                    if g_verbosity:
                        print('\t\tQuery Boot Grub Conf - End')
                    return
                self.os[i] = distro.split("'")[0]
                distro = distro.strip()
                distro_kernel = []
                if "linux /" in distro:
                    distro_kernel = distro.split("linux /")
                elif "linux\t/" in distro:
                    distro_kernel = distro.split("linux\t/")
                elif "linux16 /" in distro:
                    distro_kernel = distro.split("linux16 /")
                elif "linux16\t/" in distro:
                    distro_kernel = distro.split("linux16\t/")
                if (len(distro_kernel) > 1):
                    distro_kernel = distro_kernel[1].split()[0].split("-")
                    distro_kernel.pop(0)
                    self.kernel[i] = '-'.join(distro_kernel)
                i += 1

        while i < MAX_OS:
            self.os[i] = NOT_PRESENT
            self.kernel[i] = NOT_PRESENT
            i += 1
        if g_verbosity:
            print('\t\tQuery Boot Grub Conf - End')


class OS_Info:
    def __init__(self, hostname):
        if g_verbosity:
            print('\tQuery Operating Systems and Kernels - Start')
        self.os = [NOT_AVAILABLE, NOT_AVAILABLE, NOT_AVAILABLE]
        self.kernel = [NOT_AVAILABLE, NOT_AVAILABLE, NOT_AVAILABLE]

        if g_operating_system == 'linux':
            self.boot_grub_conf = BOOT_GRUB_CONF()
            self.os = self.boot_grub_conf.os
            self.kernel = self.boot_grub_conf.kernel
        if g_verbosity:
            print('\tQuery Operating Systems and Kernels - End')

# ----------------------------------------------------------


class Model:
    def __init__(self):
        if g_verbosity:
            print('\tQuery Model - Start')
        self.model = NOT_AVAILABLE

        stat, stdout, _ = exec_command(DMIDECODE_S_SYSTEM_PRODUCT_NAME)
        if stat == 0:
            tmp_model = stdout.splitlines()
            if len(tmp_model) > 1:
                self.model = tmp_model[-1]
            else:
                self.model = tmp_model[0]
        if g_verbosity:
            print('\tQuery Model - End')

# ----------------------------------------------------------


class Net_Info:
    def __init__(self, hostname):
        if g_verbosity:
            print('\tQuery IP and MAC - Start')
        self.net_ip = NOT_AVAILABLE
        self.net_mac = NOT_AVAILABLE
        stat, stdout, _ = exec_command(IP_A_S)
        if stat == 0:
            ip_a_s = stdout
            if '2: ' in ip_a_s:
                target_ip_a_s = ip_a_s.split('2: ')[1]
                if 'inet ' in target_ip_a_s:
                    self.net_ip = target_ip_a_s.split('inet ')[1].split()[0]
                if len(target_ip_a_s.splitlines()) > 2 and \
                        len(target_ip_a_s.splitlines()[1].split()) > 2:
                    self.net_mac = target_ip_a_s.splitlines()[1].split()[1]
        if g_verbosity:
            print('\tQuery IP and MAC - End')

# ----------------------------------------------------------


class OEM:
    def __init__(self):
        if g_verbosity:
            print('\tQuery OEM - Start')
        self.oem = NOT_AVAILABLE

        stat, stdout, _ = exec_command(DMIDECODE_S_SYSTEM_MANUFACTURER)
        if stat == 0:
            self.oem = stdout.splitlines()[-1].strip().upper()
            if self.oem.split()[0].upper() not in OEM_collection:
                self.oem = "NON-OEM"
        if g_verbosity:
            print('\tQuery OEM - End')

# ----------------------------------------------------------


class Serial_Number:
    def __init__(self):
        if g_verbosity:
            print('\tQuery Serial Number - Start')
        self.serial_number = NOT_AVAILABLE

        stat, stdout, _ = exec_command(DMIDECODE_S_SYSTEM_SERIAL_NUMBER)
        if stat == 0:
            self.serial_number = stdout.splitlines()[-1]
        if g_verbosity:
            print('\tQuery Serial Number - End')

# ----------------------------------------------------------


class HCA_PN:
    def __init__(self):
        if g_verbosity:
            print('\t\tQuery HCA Part Numbers - Start')
        self.hca_pn = [NOT_AVAILABLE, NOT_AVAILABLE, NOT_AVAILABLE]

        stat, stdout, _ = exec_command(
            "mst start; mst status; for interface in "
            "`mst status | grep ^/ | awk '{print $1}' | grep -v pci_ | "
            "grep /dev/mst/mt`; do flint -d $interface dc | "
            "grep Name; printf " + '"' + "\n" + '"' + "; done")

        i = 0
        if stat == 0:
            hca_pn_list = stdout.splitlines()
            for hca in hca_pn_list:
                if i > 2:
                    if g_verbosity:
                        print('\t\tQuery HCA Part Numbers - End')
                    return
                if "Name" in hca:
                    if " " in hca:
                        self.hca_pn[i] = hca.split()[-1]
                    else:
                        self.hca_pn[i] = hca
                    i += 1
        else:
            stat, stdout, _ = exec_command("lspci | grep Mell")
            if stat == 0:
                hca_pn_list = stdout.splitlines()
                for hca in hca_pn_list:
                    if i > 2:
                        if g_verbosity:
                            print('\t\tQuery HCA Part Numbers - End')
                        return
                    self.hca_pn[i] = hca
                    if "Mellanox Technologies " in hca:
                        self.hca_pn[i] = hca.split("Mellanox Technologies ")[1]
                        if "[" in self.hca_pn[i]:
                            self.hca_pn[i] = \
                                self.hca_pn[i].split("[")[1].split("]")[0]
                    i += 1

        while i < 3:
            self.hca_pn[i] = NOT_PRESENT
            i += 1
        if g_verbosity:
            print('\t\tQuery HCA Part Numbers - End')

# ----------------------------------------------------------


class HCA_FW_PSID:
    def __init__(self):
        if g_verbosity:
            print('\t\tQuery HCA PSID - Start')
        # self.hca_sw = [NOT_AVAILABLE, NOT_AVAILABLE, NOT_AVAILABLE]
        self.hca_fw = [NOT_AVAILABLE, NOT_AVAILABLE, NOT_AVAILABLE]
        self.hca_psid = [NOT_AVAILABLE, NOT_AVAILABLE, NOT_AVAILABLE]

        stat, stdout, _ = exec_command(
            "mst start; mst status; for interface in "
            "`mst status | grep ^/ | awk '{print $1}' | grep -v pci_ | "
            "grep /dev/mst/mt`; do flint -d $interface q; done")
        if stat == 0:
            interfaces_info = stdout
            self.hca_fw = self.retrieve_val(interfaces_info, 'FW Version:')
            self.hca_psid = self.retrieve_val(interfaces_info, 'PSID:')
        if g_verbosity:
            print('\t\tQuery HCA PSID - End')

    def retrieve_val(self, full_txt, deli):
        full_txt = full_txt.split(deli)
        full_txt.pop(0)
        triple = [NOT_PRESENT, NOT_PRESENT, NOT_PRESENT]
        for i in range(min(len(full_txt), MAX_HCA)):
            triple[i] = full_txt[i].splitlines()[0].strip()
        return triple

# ----------------------------------------------------------


class PCIE_NUMBER_TYPE_SPEED_WIDTH:
    def __init__(self):

        self.pcie_number = [NOT_AVAILABLE, NOT_AVAILABLE, NOT_AVAILABLE]
        self.pcie_type = [NOT_AVAILABLE, NOT_AVAILABLE, NOT_AVAILABLE]
        self.pcie_speed_cap = [NOT_AVAILABLE, NOT_AVAILABLE, NOT_AVAILABLE]
        self.pcie_speed = [NOT_AVAILABLE, NOT_AVAILABLE, NOT_AVAILABLE]
        self.pcie_width_cap = [NOT_AVAILABLE, NOT_AVAILABLE, NOT_AVAILABLE]
        self.pcie_width = [NOT_AVAILABLE, NOT_AVAILABLE, NOT_AVAILABLE]

        stat, stdout, _ = exec_command("lspci -d 15b3: -vvv")
        if stat != 0:
            if g_verbosity:
                print('-E- Failed to run lspci')
            return
        pcie_info = stdout
        self.pcie_number, self.pcie_type = \
            self.retrieve_number_and_type(pcie_info)
        if g_verbosity:
            print('\t\tQuery PCIE Speed Capability - Start')
        self.pcie_speed_cap = self.retrieve_val(pcie_info, 'LnkCap:', 'Speed ')
        if g_verbosity:
            print('\t\tQuery PCIE Speed Capability - End')
            print('\t\tQuery PCIE Speed - Start')
        self.pcie_speed = self.retrieve_val(pcie_info, 'LnkSta:', 'Speed ')
        if g_verbosity:
            print('\t\tQuery PCIE Speed - End')
            print('\t\tQuery PCIE Width Capability - Start')
        self.pcie_width_cap = self.retrieve_val(pcie_info, 'LnkCap:', 'Width ')
        if g_verbosity:
            print('\t\tQuery PCIE Width Capability - End')
            print('\t\tQuery PCIE Width - Start')
        self.pcie_width = self.retrieve_val(pcie_info, 'LnkSta:', 'Width ')
        if g_verbosity:
            print('\t\tQuery PCIE Width - End')

    def retrieve_val(self, full_txt, deli1, deli2):
        full_txt = full_txt.split(deli1)
        full_txt.pop(0)
        triple = [NOT_PRESENT, NOT_PRESENT, NOT_PRESENT]
        for i in range(min(len(full_txt), MAX_PCIE)):
            line = full_txt[i].splitlines()[0].strip()
            if deli2 in line:
                triple[i] = line.split(deli2)[1].split(',')[0]
        return triple

    def retrieve_number_and_type(self, full_txt):
        full_txt = full_txt.strip().splitlines()
        i = 0
        j = 0
        triple_number = [NOT_PRESENT, NOT_PRESENT, NOT_PRESENT]
        triple_type = [NOT_PRESENT, NOT_PRESENT, NOT_PRESENT]
        for line in full_txt:
            if line.startswith('0') and i < 3:
                triple_number[i] = line.split()[0]
                i += 1
            if j < 3 and ('Mellanox Technologies ' in line) \
                    and not ('Subsystem:' in line):
                triple_type[j] = 'Mellanox Technologies ' + \
                    line.split('Mellanox Technologies ')[1]
                j += 1
        return triple_number, triple_type

# ----------------------------------------------------------


class HCA_DRIVER_SW_FW:
    def __init__(self, pcie_number):
        if g_verbosity:
            print('\t\tQuery HCA DRIVER, SW And FW - Start')
        self.hca_driver = [NOT_PRESENT, NOT_PRESENT, NOT_PRESENT]
        self.hca_sw = [NOT_PRESENT, NOT_PRESENT, NOT_PRESENT]
        self.hca_fw = [NOT_PRESENT, NOT_PRESENT, NOT_PRESENT]
        i = 0
        for number in pcie_number:
            if number != NOT_AVAILABLE and number != NOT_QUERIED:
                stat, stdout, _ = exec_command(
                    "ethtool -i `ls -lt /sys/class/net/ | grep " + number +
                    " | head -1 | awk -F '/' '{ print $NF }'`")
                if stat == 0:
                    ethtool = stdout.strip()
                    if 'driver: ' in ethtool:
                        self.hca_driver[i] = ethtool.split(
                            'driver: ')[1].splitlines()[0]
                    if 'firmware-version: ' in ethtool:
                        self.hca_fw[i] = ethtool.split(
                            'firmware-version: ')[1].splitlines()[0]
                    if 'version: ' in ethtool:
                        self.hca_sw[i] = ethtool.split(
                            'version: ')[1].splitlines()[0]
                    i += 1
        if g_verbosity:
            print('\t\tQuery HCA DRIVER, SW And FW - End')

###########################################################


class Device:
    """Device could represent Server/Switch[Managed|Unmanaged]/VM/ESXi"""

    def __init__(self, hostname, subnet_manager):
        self.hostname = hostname
        self.subnet_manager = subnet_manager
        self.bios = NOT_QUERIED
        self.cpu = NOT_QUERIED
        self.kernel = [NOT_QUERIED, NOT_QUERIED, NOT_QUERIED]
        self.model = NOT_QUERIED
        self.net_ip = NOT_QUERIED
        self.net_mac = NOT_QUERIED
        self.oem = NOT_QUERIED
        self.os = [NOT_QUERIED, NOT_QUERIED, NOT_QUERIED]
        self.pcie_gen_cap = NOT_QUERIED
        self.serial_number = NOT_QUERIED
        self.hca_driver = [NOT_QUERIED, NOT_QUERIED, NOT_QUERIED]
        self.hca_pn = [NOT_QUERIED, NOT_QUERIED, NOT_QUERIED]
        self.hca_sw = [NOT_QUERIED, NOT_QUERIED, NOT_QUERIED]
        self.hca_fw = [NOT_QUERIED, NOT_QUERIED, NOT_QUERIED]
        self.hca_psid = [NOT_QUERIED, NOT_QUERIED, NOT_QUERIED]
        self.pcie_type = [NOT_QUERIED, NOT_QUERIED, NOT_QUERIED]
        self.pcie_speed_cap = [NOT_QUERIED, NOT_QUERIED, NOT_QUERIED]
        self.pcie_speed = [NOT_QUERIED, NOT_QUERIED, NOT_QUERIED]
        self.pcie_width_cap = [NOT_QUERIED, NOT_QUERIED, NOT_QUERIED]
        self.pcie_width = [NOT_QUERIED, NOT_QUERIED, NOT_QUERIED]

    def query_device_info(self):
        # Linux OS Type
        if g_operating_system == 'linux':
            try:
                if g_verbosity:
                    print('Query Hostname ' + self.hostname +
                          ' Information - Start')

                self.bios = Bios().bios

                cpu_info = CPU()
                self.cpu = cpu_info.cpu
                self.pcie_gen_cap = cpu_info.pcie_gen_cap

                os_info = OS_Info(self.hostname)
                self.os = os_info.os
                self.kernel = os_info.kernel

                self.model = Model().model
                self.oem = OEM().oem

                if self.net_ip == NOT_AVAILABLE or \
                        self.net_ip == NOT_QUERIED or \
                        self.net_mac == NOT_AVAILABLE or \
                        self.net_mac == NOT_QUERIED:
                    net_info = Net_Info(self.hostname)
                    if self.net_ip == NOT_AVAILABLE or \
                            self.net_ip == NOT_QUERIED:
                        self.net_ip = net_info.net_ip
                    if self.net_mac == NOT_AVAILABLE or \
                            self.net_mac == NOT_QUERIED:
                        self.net_mac = net_info.net_mac
                self.serial_number = Serial_Number().serial_number

                if g_verbosity:
                    print('\tQuery PCIE Information - Start')
                pcie_info = PCIE_NUMBER_TYPE_SPEED_WIDTH()
                self.pcie_number = pcie_info.pcie_number
                self.pcie_type = pcie_info.pcie_type
                self.pcie_speed_cap = pcie_info.pcie_speed_cap
                self.pcie_speed = pcie_info.pcie_speed
                self.pcie_width_cap = pcie_info.pcie_width_cap
                self.pcie_width = pcie_info.pcie_width
                if g_verbosity:
                    print('\tQuery PCIE Information - End')

                if g_verbosity:
                    print('\tQuery HCA Information - Start')
                hca_driver_sw_fw = HCA_DRIVER_SW_FW(self.pcie_number)
                self.hca_driver = hca_driver_sw_fw.hca_driver
                self.hca_sw = hca_driver_sw_fw.hca_sw
                self.hca_fw = hca_driver_sw_fw.hca_fw

                self.hca_pn = HCA_PN().hca_pn

                hca_fw_psid = HCA_FW_PSID()
                # self.hca_fw = hca_info.hca_fw
                self.hca_psid = hca_fw_psid.hca_psid
                if g_verbosity:
                    print('\tQuery HCA Information - End')
                if self.subnet_manager.lid == NOT_QUERIED:
                    if g_verbosity:
                        print('\tQuery Subnet Manager Information - Start')
                    self.subnet_manager.update()
                    if g_verbosity:
                        print('\tQuery Subnet Manager Information - End')
                if g_verbosity:
                    print('Query Hostname ' + self.hostname +
                          ' Information - End')
                    print('-' * 40)
            except Exception:
                raise

        return True


class SM():
    """SM stands for subnet manager
    """
    def __init__(self):
        self.lid = NOT_QUERIED
        self.type = NOT_QUERIED
        self.hostname = NOT_QUERIED
        self.conf = NOT_QUERIED

    def update(self):
        self.lid = NOT_AVAILABLE
        stat, stdout, _ = exec_command('sminfo')
        if stat == 0:
            try:
                self.lid = stdout.split('sm lid ')[1].split()[0]
            except:
                self.lid = NOT_AVAILABLE
        else:
            self.lid = NOT_AVAILABLE

        if self.lid == NOT_AVAILABLE:
            self.lid = NOT_AVAILABLE
            self.hostname = NOT_AVAILABLE
            self.type = NOT_AVAILABLE
            return False

        stat, stdout, _ = exec_command('smpquery ND ' + self.lid)
        if stat == 0:
            try:
                self.hostname = stdout.split('.')[-1].split()[0]
                if self.type != 'switch':
                    # Equivalent to 'if self.type == NOT_QUERIED:'
                    stat, stdout, _ = exec_command(
                        'cat /opt/opensm/opensm.conf')
                    if stat == 0:
                        self.type = 'server'
                        self.conf = stdout
                    else:
                        stat, stdout, _ = exec_command(
                            'cat /opt/ufm/files/conf/opensm/opensm.conf')
                        if stat == 0:
                            self.type = 'ufm'
                            self.conf = stdout
                        else:
                            self.type = NOT_AVAILABLE
                            return False
            except Exception:
                # print e
                if self.hostname == NOT_QUERIED:
                    self.hostname = NOT_AVAILABLE
                if self.type == NOT_QUERIED:
                    self.type = NOT_AVAILABLE
                return False
        return True


###########################################################
#            Printing Handlers
def print_device_info(d, sm, file_writer):
    file_writer.write("Server Information:\n")
    file_writer.write("hostname:\t\t%s\n" % d.hostname)
    file_writer.write("bios:\t\t\t%s\n" % d.bios)
    file_writer.write("cpu:\t\t\t%s\n" % d.cpu)
    file_writer.write("kernel1:\t\t%s\n" % d.kernel[0])
    file_writer.write("kernel2:\t\t%s\n" % d.kernel[1])
    file_writer.write("kernel3:\t\t%s\n" % d.kernel[2])
    file_writer.write("model:\t\t\t%s\n" % d.model)
    file_writer.write("net_ip:\t\t\t%s\n" % d.net_ip)
    file_writer.write("net_mac:\t\t%s\n" % d.net_mac)
    file_writer.write("oem:\t\t\t%s\n" % d.oem)
    file_writer.write("os1:\t\t\t%s\n" % d.os[0])
    file_writer.write("os2:\t\t\t%s\n" % d.os[1])
    file_writer.write("os3:\t\t\t%s\n" % d.os[2])
    file_writer.write("pcie_gen_cap:\t\t%s\n" % d.pcie_gen_cap)
    file_writer.write("serial_number:\t\t%s\n" % d.serial_number)
    file_writer.write("hca_driver1:\t\t%s\n" % d.hca_driver[0])
    file_writer.write("hca_sw1:\t\t%s\n" % d.hca_sw[0])
    file_writer.write("hca_pn1:\t\t%s\n" % d.hca_pn[0])
    file_writer.write("hca_fw1:\t\t%s\n" % d.hca_fw[0])
    file_writer.write("hca_psid1:\t\t%s\n" % d.hca_psid[0])
    file_writer.write("hca_driver2:\t\t%s\n" % d.hca_driver[1])
    file_writer.write("hca_sw2:\t\t%s\n" % d.hca_sw[1])
    file_writer.write("hca_pn2:\t\t%s\n" % d.hca_pn[1])
    file_writer.write("hca_fw2:\t\t%s\n" % d.hca_fw[1])
    file_writer.write("hca_psid2:\t\t%s\n" % d.hca_psid[1])
    file_writer.write("hca_driver3:\t\t%s\n" % d.hca_driver[2])
    file_writer.write("hca_sw3:\t\t%s\n" % d.hca_sw[2])
    file_writer.write("hca_pn3:\t\t%s\n" % d.hca_pn[2])
    file_writer.write("hca_fw3:\t\t%s\n" % d.hca_fw[2])
    file_writer.write("hca_psid3:\t\t%s\n" % d.hca_psid[2])
    file_writer.write("pcie_type1:\t\t%s\n" % d.pcie_type[0])
    file_writer.write("pcie_speed_cap1:\t%s\n" % d.pcie_speed_cap[0])
    file_writer.write("pcie_speed1:\t\t%s\n" % d.pcie_speed[0])
    file_writer.write("pcie_width_cap1:\t%s\n" % d.pcie_width_cap[0])
    file_writer.write("pcie_width1:\t\t%s\n" % d.pcie_width[0])
    file_writer.write("pcie_type2:\t\t%s\n" % d.pcie_type[1])
    file_writer.write("pcie_speed_cap2:\t%s\n" % d.pcie_speed_cap[1])
    file_writer.write("pcie_speed2:\t\t%s\n" % d.pcie_speed[1])
    file_writer.write("pcie_width_cap2:\t%s\n" % d.pcie_width_cap[1])
    file_writer.write("pcie_width2:\t\t%s\n" % d.pcie_width[1])
    file_writer.write("pcie_type3:\t\t%s\n" % d.pcie_type[2])
    file_writer.write("pcie_speed_cap3:\t%s\n" % d.pcie_speed_cap[2])
    file_writer.write("pcie_speed3:\t\t%s\n" % d.pcie_speed[2])
    file_writer.write("pcie_width_cap3:\t%s\n" % d.pcie_width_cap[2])
    file_writer.write("pcie_width3:\t\t%s\n" % d.pcie_width[2])
    file_writer.write("\nSubnet Manager Information:\n")
    file_writer.write("hostname:\t\t%s\n" % sm.hostname)
    file_writer.write("lid:\t\t\t%s\n" % sm.lid)
    file_writer.write("type:\t\t\t%s\n" % sm.type)
    if sm.type != NOT_AVAILABLE:
        file_writer.write("===== opensm.conf Start =====\n")
        file_writer.write(sm.conf)
        file_writer.write("\n")
        file_writer.write("===== opensm.conf   End =====\n")


def get_server_info(verbosity=False, out_file=None, operating_system='linux'):
    global g_verbosity
    global g_operating_system
    try:
        g_verbosity = verbosity
        g_operating_system = operating_system
        if out_file:
            file_writer = open(out_file, "w")
        else:
            file_writer = sys.stdout
        subnet_manager = SM()
        hostname = socket.gethostname()
        d = Device(hostname, subnet_manager)
        if d.query_device_info():
            print_device_info(d, subnet_manager, file_writer)
    except Exception as e:
        # print "-E-", str(e)
        return 1
    return 0


if __name__ == '__main__':
    parser = optparse.OptionParser()
    parser.add_option('-o', '--outfile',
                      help='out file name')
    parser.add_option('-q', '--verbose',
                      help='print debug information to the screen '
                      '[default False]', action='store_true', default=False)
    parser.add_option('-s', '--operating_system',
                      help='operating system of the servers to query '
                      '(linux/windows/esxi) [default linux]', default='linux')
    parser.add_option('-v', '--version',
                      help='print tool version and exit')
    options, _ = parser.parse_args()
    if options.version:
        print('Version: ' + version)
        sys.exit(0)

    get_server_info(options.verbose, options.outfile, options.operating_system)
