# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# Python Imports
import json
import os
import re

# NeoHost IFC Imports
from path_provider import MPathProvider
import neohost_plugin_ifc as ACI
from cmd_exec import Command
from request_factory import RequestFactory
from msg_mgr import MsgMgr, MsgConstants

# Plugin Imports
from mft_fw_defs import MftFwDefs
import mft_fw_exceptions
from neohost_common import NeoHostCommon
#from common_help_func import CommonHelpFuncs


class BaseDeviceResetCmd(ACI.AbsNeoHostCommandIFC):
    MLX_FW_RESET_CMD = "mlxfwreset"
    devMstLine = ""

    def _getNameByPCIDeviceId(self, pciId):
        # TODO - move to common
        dev_name = ""
        cmd = "mst status -v"
        rc, out, _ = Command(cmd).execute()
        if not rc:
            lines = out.splitlines()
            for line in lines:
                if line.find(pciId) != -1:
                    self.devMstLine = line
                    parts = line.split()
                    if len(parts) == 2:
                        dev_name = parts[0]
                    break
        return dev_name

    def _getDevUid(self, req):
        devUid = req["devUid"]
        if NeoHostCommon.isWindowsOs():
            parts = devUid.split(":")
            if len(parts[0]) == 4 and len(parts) == 3:
                devUid = "%s:%s" % (parts[1], parts[2])
            dev_name = self._getNameByPCIDeviceId(devUid)
            devUid = dev_name 
            if dev_name == "":
                raise mft_fw_exceptions.MftFwException("Unknown device - %s" % devUid)
        elif NeoHostCommon.isFreebsdOs():
            devUid = NeoHostCommon.PCIDevIdToFreebsdFormat(devUid)
        else:
            pattern = "0000:"
            if devUid.startswith(pattern):
                devUid = devUid[len(pattern):]

        return devUid

    def _getCmd(self, devUid, req):
        pass

    def _parseOutput(self, out):
        pass

    def _formatErrMessage(self, rc, err):
        pass

    def _getMlxFwResetToolPath(self):
        root_dir = MPathProvider().get_root_dir()
        tools_path = os.path.join(root_dir, MPathProvider.PLUGINS_DIR,
                            MftFwDefs.MFT_FW_PLUGIN_NAME,
                            MftFwDefs.MFT_FW_TOOLS_DIR)
        return os.path.join(tools_path, self.MLX_FW_RESET_CMD)

    def execute_command(self, json_request, exec_opt):
        req = json.loads(json_request)
        devUid = self._getDevUid(req)

        #TODO - check device supported

        cmd = self._getCmd(devUid, req)
        if cmd is None:
            msg = self._formatErrMessage(44, "Can't apply fw reset for BlueField")
            raise mft_fw_exceptions.MftFwException(msg)
        rc, out, err = Command(cmd).execute()
        if rc != 0:
            msg = self._formatErrMessage(rc, err or out)
            raise mft_fw_exceptions.MftFwException(msg)
        result = self._parseOutput(out)
        return MftFwDefs.MFT_FW_STATUS_SUCCESS, json.dumps(result)


class GetDeviceResetLevel(BaseDeviceResetCmd):
    def __init__(self):
        super(GetDeviceResetLevel, self).__init__(
            cmd_name="GetDeviceResetLevel",
            cmd_desc="Get minimal reset level that requires to reload",
            cmd_type=ACI.EnumCmdType.Cmd_Type_Get,
            cmd_scope=ACI.EnumCmdScope.Cmd_Scope_Device,
            supp_exec_mask=ACI.EnumCmdExecMode.Cmd_Exec_Mode_All,
            cmd_in_desc="devUid - device UID",
            cmd_out_desc="minimal reset level",
            extra_str="")

    def _getCmd(self, devUid, req):
        if NeoHostCommon.isWindowsOs() or NeoHostCommon.isFreebsdOs():
            res = "mlxfwreset -d %s q" % devUid
        else:
            res = "%s -d %s q" % (self._getMlxFwResetToolPath(), devUid)
        return res

    def _formatErrMessage(self, rc, err):
        if rc == 127:
            err_message = "Cannot get reset device level, MFT is not "\
                "installed, please reset machine instead"
        else:
            err_message = "Failed to get device reset level: %s" % err
        return err_message

    def _parseOutput(self, out):
        result_dict = dict()
        lineParser = re.compile(r"(\d+):\s+(.*\S)\s+\-Supported\s+\(default\)")
        lineObject = None
        level = None
        desc = None
        for line in out.splitlines():
            line = line.strip()
            lineObject = lineParser.match(line)
            if not lineObject:
                continue
            level, desc = lineObject.group(1,2)
            if  level and desc:
                break
        result_dict["level"] = int(level)
        result_dict["description"] = desc.strip()
        return result_dict


class SetDeviceReset(BaseDeviceResetCmd):
    def __init__(self):
        super(SetDeviceReset, self).__init__(
            cmd_name="SetDeviceReset",
            cmd_desc="Perform a FW reset on device to (re)load FW",
            cmd_type=ACI.EnumCmdType.Cmd_Type_Get,
            cmd_scope=ACI.EnumCmdScope.Cmd_Scope_Device,
            supp_exec_mask=ACI.EnumCmdExecMode.Cmd_Exec_Mode_Sync,
            cmd_in_desc="devUid - device UID, reset level",
            cmd_out_desc="Success",
            extra_str="")

    def _getCmd(self, devUid, req):
        level = req["level"]
        super(SetDeviceReset, self)._getNameByPCIDeviceId(devUid)
        if NeoHostCommon.isBlueFieldDevice(self.devMstLine):
            return None
        else:
            if NeoHostCommon.isWindowsOs() or NeoHostCommon.isFreebsdOs():
                cmd = "mlxfwreset -d %s --level %s --no_mst_restart --yes reset" % \
                     (devUid, level)
            else:
                cmd = "%s -d %s --level %s --no_mst_restart --yes reset" % \
                      (self._getMlxFwResetToolPath(), devUid, level)
        return cmd

    def _formatErrMessage(self, rc, err):
        if rc == 127:
            err_message = "Cannot reset device, MFT is not installed, "\
                "please reset machine instead"
        else:
            err_message = "Failed to reset device: %s" % err
        return err_message

    def _parseOutput(self, out):
        return "Success"
