# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# @author: Simon Raviv
# @date: January 09, 2018

from abc import ABCMeta
from numbers import Real

from performance.common.units_measure import UnitsMeasure
from performance.utilities.utils import Utils


class ABCAttribute(object):
    """ Abstract performance attribute.
    """

    __metaclass__ = ABCMeta

    def __init__(self, name='', description='',
                 units_measure=UnitsMeasure.NONE, timestamp=0, value=0):
        super(ABCAttribute, self).__init__()
        self._name = name
        self._description = description
        self._units_measure = units_measure
        self._timestamp = timestamp
        self._value = value

    @property
    def name(self):
        """ Get attribute's name.
        """
        return self._name

    @property
    def description(self):
        """ Get attribute's description.
        """
        return self._description

    @description.setter
    def description(self, description):
        """ Set attribute's description.
        """
        self._description = description

    @property
    def units_measure(self):
        """ Get attribute's units measure.
        """
        return self._units_measure

    @units_measure.setter
    def units_measure(self, units_measure):
        """ Set attribute's units measure.
        """
        Utils.one_of(units_measure, vars(UnitsMeasure).values())
        self._units_measure = units_measure

    @property
    def timestamp(self):
        """ Get attribute's time stamp.
        """
        return self._timestamp

    @timestamp.setter
    def timestamp(self, timestamp):
        """ Set attribute's times tamp.
        """
        self._timestamp = timestamp

    @property
    def value(self):
        """ Get attribute's value.
        """
        return self._value

    @value.setter
    def value(self, value):
        """ Set attribute's value.
        """
        self._value = int(value) if int(value) == value else round(value, 4)

    def __add__(self, other):
        """ Implements plus operator overloading.
        """
        if isinstance(other, ABCAttribute):
            return ABCAttribute(value=(self.value + other.value))
        raise NotImplementedError("__add__", ABCAttribute.__name__)

    def __sub__(self, other):
        """ Implements minus operator overloading.
        """
        if isinstance(other, ABCAttribute):
            return ABCAttribute(value=(self.value - other.value))
        raise NotImplementedError("__sub__", ABCAttribute.__name__)

    def __gt__(self, other):
        """ Implements grater than operator overloading.
        """
        if isinstance(other, ABCAttribute):
            return self.value > other.value
        if isinstance(other, Real):
            return self.value > other
        raise NotImplementedError("__gt__", ABCAttribute.__name__)

    def __div__(self, other):
        """ Implements divide operator overloading.
        """
        if isinstance(other, ABCAttribute):
            return self.value / float(other.value)
        if isinstance(other, Real):
            return self.value / float(other)
        raise NotImplementedError("__div__", ABCAttribute.__name__)

    def __mul__(self, other):
        """ Implements multiply operator overloading.
        """
        if isinstance(other, ABCAttribute):
            return self.value * other.value
        if isinstance(other, Real):
            return self.value * other
        raise NotImplementedError("__mul__", ABCAttribute.__name__)
