# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Simon Raviv
# @date: December 24, 2017

from performance.common.units_measure import UnitsMeasure
from performance.common.units import Units
from performance.common.utilization_reference import UtilizationReference
from performance.common.counters_names import CountersNames
from performance.common.utilization_reference import UtilizationReference \
    as UtilReference
from performance.entities.abc_attribute import ABCAttribute
from performance.utilities.utils import Utils
from performance.services.mcra.connectcx_counters_map import \
    CountersNamesDescriptionMap as NamesMap


class Counter(ABCAttribute):
    """ Represents MCRA chip counter.
    """
    def __init__(self, value=0, unit=Units.NONE, selector=None, name='',
                 units=UnitsMeasure.NONE,
                 utilization_reference=UtilizationReference.NONE,
                 utilization_bad=UtilizationReference.NONE,
                 utilization_good=UtilizationReference.NONE):
        super(Counter, self).__init__(units_measure=units)
        self._name = name
        self._description = NamesMap().get_description(self.name)
        self._unit = unit
        self._selector = selector
        self.__set_unit_index()
        self._utilization_reference = utilization_reference
        self._utilization_bad = utilization_bad
        self._utilization_good = utilization_good
        self._utilization_percentage = 0
        self.set_value(value=value)

    @ABCAttribute.name.setter
    def name(self, name):
        """ Set counter name.
        """
        Utils.one_of(name, vars(CountersNames).values())
        self._name = name

    @property
    def unit(self):
        """ Get counter unit.
        """
        return self._unit

    @unit.setter
    def unit(self, unit):
        """ Set counter unit.
        """
        self._unit = unit

    @property
    def selector(self):
        """ Get counter selector.
        """
        return self._selector

    @selector.setter
    def selector(self, selector):
        """ Set counter selector.
        """
        self._selector = selector

    @property
    def utilization_reference(self):
        """ Get counter utilization reference.
        """
        return self._utilization_reference

    @utilization_reference.setter
    def utilization_reference(self, utilization_reference):
        """ Set counter utilization reference.
        """
        self._utilization_reference = utilization_reference

    @property
    def utilization_bad(self):
        """ Get counter utilization bad.
        """
        return self._utilization_bad

    @utilization_bad.setter
    def utilization_bad(self, utilization_bad):
        """ Set counter utilization bad.
        """
        self._utilization_bad = utilization_bad

    @property
    def utilization_good(self):
        """ Get counter utilization good.
        """
        return self._utilization_good

    @utilization_good.setter
    def utilization_good(self, utilization_good):
        """ Set counter utilization good.
        """
        self._utilization_good = utilization_good

    @property
    def utilization_percentage(self):
        """ Get counter utilization percentage.
        """
        return self._utilization_percentage

    @utilization_percentage.setter
    def utilization_percentage(self, utilization_percentage):
        """ Set counter utilization percentage.
        """
        self._utilization_percentage = utilization_percentage

    def set_value(self, one_sec_correction=None, value=None):
        """ Sets counter value.
        """
        if one_sec_correction:
            self._value = int(self.unit.get_value(self.unit_index) /
                              one_sec_correction)
        elif value:
            self._value = int(value)

    def get_unit_name(self):
        """ Returns counter's unit name.
        """
        return self.unit.name

    def set_reference(self, reference=None):
        """ Sets counter utilization reference.
        """
        if self.utilization_reference == UtilReference.NONE \
                or reference is None:
            return
        if reference[self.utilization_reference]:
            util_reference = float(reference[self.utilization_reference])
            util_percentage = self.value / util_reference
            self.utilization_percentage = int(util_percentage * 100)

    def __set_unit_index(self):
        """ Sets counter's index in a unit.
        """
        self.unit_index = 0
        if self.selector is not None:
            self.unit_index = self.unit.add_counter(self.selector)
