# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# @author: Tal Gilboa
# @date: Mar 16, 2017

import platform
from performance.api.info_entry import InfoEntry
from performance.api.schema_keys import SchemaKeys
from performance.services.logger import PerformanceLog
from performance.common.severity import Severity


class OS(InfoEntry):
    """ Describes the operation system.
    """
    CENTOS = "CENTOS"
    RH6_4 = "RH6.4"
    RH6_5 = "RH6.5"
    RH6_6 = "RH6.6"
    RH6_7 = "RH6.7"
    RH7_0 = "RH7.0"
    RH7_1 = "RH7.1"
    RH7_2 = "RH7.2"
    RH7_3 = "RH7.3"
    FEDORA = "FEDORA"
    SLES11_3 = "SLES11.3"
    SLES11_4 = "SLES11.4"
    SLES12 = "SLES12"
    UBUNTU14_04 = "UBUNTU14.04"
    UBUNTU14_10 = "UBUNTU14.10"
    UBUNTU15_04 = "UBUNTU15.04"
    UBUNTU15_10 = "UBUNTU15.10"
    UBUNTU16_04 = "UBUNTU16.04"
    UBUNTU16_10 = "UBUNTU16.10"
    ESX = "ESX"
    WIN = "WIN"
    FREEBSD = "FREEBSD"
    POWER_KVM = "POWER KVM"
    SUNOS = "SUNOS"
    UEFI = "UEFI"
    DEBIAN = "DEBIAN"
    UNKNOWN = "UNKNOWN"
    ALL = "ALL"
    SUPPORTED_OS = [
                      RH6_4, RH6_5, RH6_6, RH6_7, RH7_0, RH7_1, RH7_2,
                      SLES11_3, SLES11_4, SLES12,
                      UBUNTU14_04, UBUNTU14_10, UBUNTU15_04, UBUNTU15_10,
                      UBUNTU16_04, UBUNTU16_10, CENTOS
                   ]
    SYSTEMCTL_ACCESS_OS = [RH7_0, RH7_1, RH7_2, RH7_3]
    SUSE = [SLES11_3, SLES11_4, SLES12]
    PPC_DEVICE_NEW_FORMAT_OS = [RH7_0, RH7_1, RH7_2, RH7_3]
    NO_IPV6_FORWARDING_SUPPORT_OS = [UBUNTU14_04]

    def __init__(self):
        super(OS, self).__init__(SchemaKeys.OS, False)

    def query(self):
        """ Queries the system for its OS information.
        """
        entry = InfoEntry("OS", True, in_val=self.get_os(),
                          desc="The installed OS version",
                          severity=Severity.UNDEFINED)
        self.update_value(entry)
        entry = InfoEntry("Kernel", True, in_val=self.get_kernel(),
                          desc="The installed kernel version",
                          severity=Severity.UNDEFINED)
        self.update_value(entry)

    def get_os(self):
        """ Return local OS Type (one of the list OS_ALL)
            in case of unknown system type, it's return OS.UNKNOWN.
        """
        os_platform = platform.system().lower()
        if os_platform in ["windows", "microsoft"]:
            return OS.WIN

        if os_platform == "uefi":
            return OS.UEFI

        if os_platform == "vmkernel":
            return OS.ESX

        if os_platform == "sunos":
            return OS.SUNOS

        if os_platform == "linux":
            linux_dist = platform.dist()[0].lower()
            os_version = platform.platform().split('-with-')[1].split('-')[1]
            if linux_dist == 'redhat':
                if os_version == '6.4':
                    return OS.RH6_4
                if os_version == '6.5':
                    return OS.RH6_5
                if os_version == '6.6':
                    return OS.RH6_6
                if os_version == '6.7':
                    return OS.RH6_7
                if os_version == '7.0':
                    return OS.RH7_0
                if os_version == '7.1':
                    return OS.RH7_1
                if os_version == '7.2':
                    return OS.RH7_2
                if os_version == '7.3':
                    return OS.RH7_3
            if linux_dist == 'fedora':
                return OS.FEDORA
            if linux_dist == 'suse':
                if os_version == '11':
                    return OS.SLES11_3
                if os_version == '12':
                    return OS.SLES12
            if linux_dist == 'ubuntu':
                if os_version == '14.04':
                    return OS.UBUNTU14_04
                if os_version == '14.10':
                    return OS.UBUNTU14_10
                if os_version == '15.04':
                    return OS.UBUNTU15_04
                if os_version == '15.10':
                    return OS.UBUNTU15_10
                if os_version == '16.04':
                    return OS.UBUNTU16_04
                if os_version == '16.10':
                    return OS.UBUNTU16_10
            if linux_dist == 'centos':
                return OS.CENTOS
            if linux_dist == 'debian':
                return OS.DEBIAN
            if linux_dist == "freebsd":
                return OS.FREEBSD
            if linux_dist == "ibm_powerkvm":
                return OS.POWER_KVM

        PerformanceLog().get_log().warning("Unknown OS [%s,%s,%s]. \
                                           Tuning might be non-optimized." %
                                           (str(platform.system()),
                                            str(platform.release()),
                                            str(platform.dist())))
        return OS.UNKNOWN

    def get_kernel(self):
        """ Return kernel version.
        """
        return platform.release()

    def is_supported_system(self):
        """ Check if the local OS/ARCH is in the given
            supportedOSlist and supportedARCHs
            return True if true, Otherwise UnkownSystemError is raised.
        """
        currentOS = self.get_os()
        if currentOS in OS.SUPPORTED_OS:
            return True
        return False
