# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Simon Raviv
# @date: December 24, 2017


class Unit(object):
    """ Represents MCRA chip unit.
    """
    def __init__(self, name, en_addr, en_start_bit=0, start_addr=None,
                 selector_addr=None, size=16):
        """
        Unit class constructor.
        name: Unit's name.
        en_addr: Performance counters enabler bit address.
        en_start_bit: Performance counters enabler bit offset.
        start_addr: Performance counters start address.
        selector_addr: Performance selectors start address.
        size: Number of performance counters in performance counters section.
        """
        self.name = name
        self.en_addr = en_addr
        self.en_start_bit = en_start_bit
        if start_addr:
            self.start_addr = start_addr
        else:
            self.start_addr = en_addr + 0x20

        if selector_addr:
            self.selector_addr = selector_addr
        else:
            self.selector_addr = en_addr - 0x20
        self.last_selector_index = 0
        self.size = size
        self.selectors = [0] * (self.size / 2)
        self.values = [0] * self.size
        self.snapshot = []

    def add_counter(self, selector):
        """ Add counter to the selector.
        """
        if self.last_selector_index >= self.size - 1:
            return -1
        if self.last_selector_index % 2 == 0:
            selector = selector << 16
        self.selectors[self.last_selector_index / 2] |= selector
        self.last_selector_index += 1
        return self.last_selector_index - 1

    def get_value(self, index):
        """ Returns counter's value.
        """
        return self.values[index]

    def read_values(self, mf):
        """ Read value from the address.
        """
        self.values = mf.read_4block(self.start_addr, self.size)

    def set_selectors(self, mf):
        """ Set selectors addresses.
        """
        mf.write_4block(self.selector_addr, self.selectors)

    def reset_counters(self, mf):
        """ Reset counters.
        """
        mf.write_4block(self.start_addr, [0] * self.size)

    def take_snapshot(self, mf):
        """ Read counters from the chip.
        """
        self.snapshot = mf.read_4block(self.selector_addr, self.size / 2)

    def restore(self, mf):
        """ Restore counters from snapshot.
        """
        if self.snapshot:
            mf.write_4block(self.selector_addr, self.snapshot)
