# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Simon Raviv
# @date: April 02, 2017

import time
import subprocess
import json

import neohost_plugin_ifc as APIFC
from request_factory import RequestFactory
from msg_mgr import MsgMgr, MsgConstants
from performance_defs import PerformanceDefs
from services.workers.service_worker import WorkerService
from services.workers.feed_worker import WorkerFeed
from api.counter import Counter
from api.analysis_attribute import AnalysisAttribute
from api.performance_counters import PerformanceCounters
from api.metadata import MetaData
from api.metadata_group import MetaDataGroup
from api.analysis_metadata import AnalysisMetaData
from api.analysis_metadata_group import AnalysisMetaDataGroup
from api.schema_keys import SchemaKeys
from performance_exceptions import PerformanceException
from common.device_ids import DeviceIDs
from common.api_keys import APIKeys
from common.filter_mode import FilterMode
from common.support_condition_keys import SupportServiceKeys as SSK
from services.recording import Recording
from services.filtering.filter import Filter
from services.analysis.analysis import Analysis
from services.support.support import Support


class GetDevicePerformanceCounters(APIFC.AbsNeoHostCommandIFC):
    """ API command for getting a device's performance counters.
        The command will query for a pre-defined set of counters
        and return an output in the format defined by the command's schema.
    """
    def __init__(self):
        super(GetDevicePerformanceCounters, self).__init__(
              cmd_name="GetDevicePerformanceCounters",
              cmd_desc="Get the device's performance counters",
              cmd_type=APIFC.EnumCmdType.Cmd_Type_Get,
              cmd_scope=APIFC.EnumCmdScope.Cmd_Scope_Device,
              supp_exec_mask=APIFC.EnumCmdExecMode.Cmd_Exec_Mode_Sync,
              cmd_in_desc="devUid: The device's PCI location, \
                          mstDevice: The device's MST name, \
                          rdmaDevice: The device's RDMA name",
              cmd_out_desc="performanceCounters: The device's performance \
                           counters information",
              extra_str="")

    def _getInterface(self, ifcUid):
        """ Get an interface to the command.
        """
        sub_req_obj = RequestFactory.createRequest(
                      1,
                      "performance",
                      "GetDevicePerformanceCounters",
                      None)
        _, rc, response = MsgMgr().handle(MsgConstants.NEOHOST_REQUEST,
                                          sub_req_obj)
        if rc != 0:
            raise PerformanceException("Failed to get device counters")

    def __get_device_info(self, request):
        """ Returns device PCI address and ID.
        """
        # Convert from PCI address to MST device:
        device_pci_address = str(request[APIKeys.DEVUID])
        device_id_command = "cat /sys/bus/pci/devices/{pci}/device".format(
            pci=device_pci_address)
        proc = subprocess.Popen(device_id_command,
                                stdin=subprocess.PIPE,
                                stdout=subprocess.PIPE,
                                stderr=subprocess.PIPE,
                                shell=True)
        output, return_code = proc.communicate()

        device_id = 0
        if output:
            device_id = int(output, 16)

        if not (device_id and output) or (device_id not in DeviceIDs.ALL):
            error = ("Failed to get device performance counters: "
                     "-E- Failed to Identify Device: {device}")
            error = error.format(device=device_pci_address)
            raise PerformanceException(error)

        return device_pci_address, device_id

    def __get_analysis_api_info(self, request, counters, analyzers, timestamp):
        """ Returns analyzers and their meta-data.
        """
        analysis_metadata = None
        sampled_analysis = None

        if APIKeys.ANALYSIS in request:
            analysis_service = Analysis(data=counters,
                                        analyzers=analyzers,
                                        timestamp=timestamp)
            analysis_service.analyse()

            sampled_analysis = list()
            for analyzer in analysis_service.analyzers:
                api_analyzer = AnalysisAttribute(analysis_attribute=analyzer)
                sampled_analysis.append(api_analyzer)

            analysis_groups_metadata = list()

            for group, analyzers in analysis_service.get_groups().iteritems():
                analysis_groups_metadata.append(
                    AnalysisMetaDataGroup(group,
                                          analyzers))

            analysis_metadata = AnalysisMetaData(analysis_groups_metadata)

        return sampled_analysis, analysis_metadata

    def __get_counters_api_info(self, counters, unit_items):
        """ Returns counters and their meta-data.
        """
        sampled_counters = list()
        for counter in counters:
            api_counter = Counter(counter=counter)
            sampled_counters.append(api_counter)

        groups_metadata = list()

        for unit, counters in unit_items:
            groups_metadata.append(MetaDataGroup(unit, counters))

        metadata = MetaData(groups_metadata)

        return sampled_counters, metadata

    def __set_counters_reference(self, service, counters):
        """ Set counter's utilization reference
        """
        reference = service.nic.get_reference_data(counters)
        sampled_counters = list()
        for counter in counters:
            counter.set_reference(reference)

    def __filter_counters(self, counters):
        """ Filter unneeded counters for presentation.
        """
        filter_service = Filter(counters)
        mode = filter_service.filter()
        if mode == FilterMode.DONT_RUN:
            raise PerformanceException("Performance plug-in is not supported")

    def __record(self, result, request):
        """ Record the result if needed.
        """
        if APIKeys.RECORD_PATH in request:
            record_service = Recording(request[APIKeys.RECORD_PATH])
            record_service.record(result)

    def __check_support(self, **support_service_parameters):
        """ Check if the command is supported, based on conditinos.
        """
        support_service = Support(support_service_parameters)
        support_service.run_inspection()

    def execute_command(self, json_request, exec_opt):
        """ Execute the get device performance counters command.
        """
        request = json.loads(json_request)
        device_pci_address, device_id = self.__get_device_info(request)

        support_service_parameters = dict()
        support_service_parameters[SSK.PCI_DEVICE] = device_pci_address
        self.__check_support(**support_service_parameters)

        feed = WorkerFeed()
        if feed.get_error():
            error = feed.get_error()
            raise PerformanceException(error)

        service = WorkerService(device_pci_address, device_id)
        unit_items = service.nic.mcra.get_counters_per_unit().items()
        timestamp = service.nic.timestamp

        if service.get_error():
            error = feed.get_error()
            raise PerformanceException(error)

        # Sampled counters data:
        service.parent_worker = feed
        feed.add_child(service)

        # Start (recursively):
        feed.start(max_iterations=1)

        # Our sampling rate is a bit less than a second, thus it is
        # waste of CPU to do polling on results, it is better to
        # sleep 0.5 second between queries. Sleep for 0.5 second
        # is enough to have only one redundant poll query.
        while feed.is_alive():
            time.sleep(0.5)

        feed.data = feed.outQ.get()
        data = feed.data
        counters = data.data

        self.__set_counters_reference(service=service, counters=counters)

        sampled_analysis, analysis_metadata = self.__get_analysis_api_info(
            request=request,
            counters=data,
            analyzers=service.nic.analyzers,
            timestamp=timestamp)

        sampled_counters, metadata = self.__get_counters_api_info(
            counters=counters, unit_items=unit_items)

        # Build results objects:
        performance_counters = PerformanceCounters(
            counters=sampled_counters,
            metadata=metadata,
            analysis=sampled_analysis,
            analysis_metadata=analysis_metadata).to_dictionary()

        data = performance_counters[SchemaKeys.PERFORMANCE_COUNTERS]
        self.__filter_counters(data)

        # Final result matching JSON scheme:
        result = json.dumps(performance_counters)
        self.__record(result=result, request=request)

        return PerformanceDefs.PERFORMANCE_STATUS_SUCCESS, result
