# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# @author: Simon Raviv
# @date: December 19, 2017

from performance.common.analyzers_groups import AnalyzersGroups
from performance.utilities.utils import Utils


class Analysis(object):
    """ Analysis service for the module,
        will include analyzers for different performance attributes.
    """

    def __init__(self, data, analyzers, timestamp):
        super(Analysis, self).__init__()
        self.__data = data
        self.__analyzers = analyzers
        self.__timestamp = timestamp
        self.__remove_unsupported_analyzers()

    def analyse(self):
        """ Run all analyzers.
        """
        for analyzer in self.__analyzers:
            analyzer.data = self.__data
            analyzer.timestamp = self.__timestamp
            analyzer.other_analyzers = self.__analyzers
            analyzer.analyse()

    @property
    def analyzers(self):
        """ Get performance analyzers.
        """
        return self.__analyzers

    def get_groups(self):
        """ Returns analyzers grouping by group types.
        """
        result = dict()

        for group in Utils.get_all(AnalyzersGroups):
            result[group] = list()

        for analyzer in self.analyzers:
            result[analyzer.group].append(analyzer.name)

        return result

    def __filter_analyzers(self, items):
        """ Filter analyzers based on the input.
            This will be used to filter by counters or analyzers.
            Note that analyzer can be dependent on counters
            and/or other analyzers.
            This algorithm works correctly, since analyzers
            dependencies stated by the order of their initialization
            in MCRACounters class abstractions.
        """
        existing_items_names = set(item.name for item in items)

        self.__analyzers = filter(
            lambda analyzer: analyzer.is_required_data_exist(
                existing_items_names),
            self.__analyzers)

    def __remove_unsupported_analyzers(self):
        """ Remove unsupported analyzers.
            In case there are missing counters or analyzers
            for an analyzer calculation,
            this method aims to remove those analyzers.
        """
        counters = self.__data.data
        analyzers = self.__analyzers
        self.__filter_analyzers(items=(counters + analyzers))
