# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# @author: Simon Raviv
# @date: December 21, 2017

from abc import ABCMeta, abstractmethod

from performance.common.analyzers_names import AnalyzersNames
from performance.common.units_measure import UnitsMeasure
from performance.common.analyzer_value_type import AnalyzerValueType
from performance.common.analyzers_groups import AnalyzersGroups
from performance.common.gui_components import GUIComponents
from performance.entities.abc_attribute import ABCAttribute
from performance.utilities.utils import Utils


class ABCAnalyzer(ABCAttribute):
    """ Abstract analyzer, all the analyzers will be derived
        from this class.
    """

    __metaclass__ = ABCMeta

    def __init__(self, data=None, other_analyzers=None, name='',
                 description='', units_measure=UnitsMeasure.NONE,
                 timestamp=0, value=0, value_type=AnalyzerValueType.NONE,
                 group=AnalyzersGroups.NONE,
                 gui_component=GUIComponents.NONE):
        super(ABCAnalyzer, self).__init__(units_measure=units_measure,
                                          timestamp=timestamp, value=value)
        self._name = name
        self._data = data
        self._other_analyzers = other_analyzers
        self._value_type = value_type
        self._group = group
        self._gui_componenet = gui_component
        self._required_counters = list()
        self._required_analyzers = list()

    @abstractmethod
    def analyse(self):
        """ Run the calculation analysis.
        """
        pass

    @ABCAttribute.name.setter
    def name(self, name):
        """ Set analyzer name.
        """
        Utils.one_of(name, vars(AnalyzersNames).values())
        self._name = name

    @property
    def value_type(self):
        """ Get analyzer value type.
        """
        return self._value_type

    @value_type.setter
    def value_type(self, value_type):
        """ Set analyzer value type.
        """
        Utils.one_of(value_type, vars(AnalyzerValueType).values())
        self._value_type = value_type

    @property
    def data(self):
        """ Get counters data.
        """
        return self._data

    @data.setter
    def data(self, data):
        """ Set counters data.
        """
        self._data = data

    @property
    def other_analyzers(self):
        """ Get other analyzers data.
        """
        return self._other_analyzers

    @other_analyzers.setter
    def other_analyzers(self, other_analyzers):
        """ Set other analyzers data.
        """
        self._other_analyzers = other_analyzers

    @property
    def group(self):
        """ Get analyzer's group.
        """
        return self._group

    @group.setter
    def group(self, group):
        """ Set analyzer's group.
        """
        Utils.one_of(group, vars(AnalyzersGroups).values())
        self._group = group

    @property
    def gui_component(self):
        """ Get analyzer's GUI component.
        """
        return self._gui_componenet

    @gui_component.setter
    def gui_component(self, gui_component):
        """ Set analyzer's GUI component.
        """
        Utils.one_of(gui_component, vars(GUIComponents).values())
        self._gui_componenet = gui_component

    def get_analyzer_by_name(self, name):
        """ Returns analyzer class instance by name.
        """
        return Utils.get_object_by_name(self.other_analyzers, name)

    def round_up_to_one(self, number):
        """ Rounds a number to one, in case lower than one.
        """
        if number < 1:
            number = 1
        return number

    def _add_required_counter(self, counter):
        """ Add to analyzer's required counters.
        """
        self._required_counters.append(counter)

    def _add_required_analyzer(self, analyzer):
        """ Add to analyzer's required analyzers.
        """
        self._required_analyzers.append(analyzer)

    def is_required_data_exist(self, existing_information):
        """ Check if all the required information for
            analyzer's calculation exist.
        """
        required_information = (self._required_counters
            + self._required_analyzers)

        for required_item in required_information:
            if required_item not in existing_information:
                existing_information.remove(self.name)
                return False
        return True
