# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# @author: Tal Gilboa
# @date: Mar 16, 2017

import re
from performance.services.status import Status
from performance.services.cpu.access import Access
from performance.services.cpu.vendor import Vendor
from performance.services.cpu.architecture import Architecture
from performance.services.cpu.vendor_constants import VendorConstants
from performance.services.command.linux_command import LinuxCommand


class AccessLinux(Access):
    """ CPU access class for Linux OS
    """

    GET_CPUINFO_CMD = "cat /proc/cpuinfo"
    GET_ARCHITECTURE_INFO_CMD = "uname -m"
    GET_LSCPU_CMD = "lscpu"
    GET_NUMA_NODES_CMD = "ls /sys/devices/system/node/ | grep node"
    GET_NUMA_CORES_CMD = "ls /sys/devices/system/node/node%s/"
    GET_ALL_CPUS_CMD = "ls /sys/devices/system/cpu/"
    GET_CORE_ID_CMD = \
        "cat /sys/devices/system/node/node%s/cpu%s/topology/core_id"
    GET_WATCHDOG_CMD = "cat /proc/sys/kernel/watchdog"
    GET_NMI_WATCHDOG_CMD = "cat /proc/sys/kernel/nmi_watchdog"
    GET_VM_STAT_INTERVAL_CMD = "cat /proc/sys/vm/stat_interval"
    DMIDECODE = "dmidecode"

    def __init__(self):
        super(AccessLinux, self).__init__()
        self.command = LinuxCommand()

    def collect_info(self):
        """ Collects CPU information from the system and
            updates the class fields
        """
        (rc, lscpu_output) = self.command.run_command_warn_when_fail(
            AccessLinux.GET_LSCPU_CMD,
            "Unable to collect cpu info.")
        (rc, cpuinfo_output) = self.command.run_command_warn_when_fail(
            AccessLinux.GET_CPUINFO_CMD, "Unable to collect cpu info.")
        (rc, architecture_info_output) = \
            self.command.run_command_warn_when_fail(
            AccessLinux.GET_ARCHITECTURE_INFO_CMD,
            "Unable to collect architecture info.")
        architecture_str = architecture_info_output.lower()
        if Architecture.X86_64.lower() in architecture_str:
            self.vendor = Vendor.INTEL
            self.collect_intel_cpu_info(cpuinfo_output, lscpu_output)
        elif Architecture.PPC64LE.lower() or Architecture.PPC64.lower() \
                in architecture_str:
            self.vendor = Vendor.IBM
            self.collect_ppc_cpu_info(cpuinfo_output)
        elif Architecture.AARCH64.lower() in architecture_str:
            self.vendor = Vendor.ARM
            self.collect_arm_cpu_info(cpuinfo_output)

        self.collect_common_cpu_info()

    def collect_common_cpu_info(self):
        """ Collects CPU information common for all architectures.
        """
        arr = []
        (rc, output) = self.command.run_command_warn_when_fail(
            AccessLinux.GET_NUMA_NODES_CMD,
            "Unable to collect NUMA node info.")
        if not rc:
            for line in output.split("\n"):
                arr.append(int(line.replace('node', '').strip()))
            self.sockets = len(arr)

        socket_dict = {}
        for socket in arr:
            socket_dict[socket] = []
            (rc, output) = self.command.run_command_warn_when_fail(
                AccessLinux.GET_NUMA_CORES_CMD % socket,
                "Unable to collect NUMA cores info.")
            if not rc:
                for element in output.split('\n'):
                    if 'cpu' in element \
                            and element.replace('cpu', '').isdigit():
                        socket_dict[socket].append(
                            int(element.replace('cpu', '')))
                socket_dict[socket] = sorted(socket_dict[socket])
        if not socket_dict:
            # If no NUMA found - consider all CPUs to be on NUMA 0.
            socket_dict[0] = []
            (rc, output) = self.command.run_command_warn_when_fail(
                AccessLinux.GET_ALL_CPUS_CMD,
                "Unable to find any CPU on the system.")
            for element in output.split('\n'):
                if 'cpu' in element and element.replace('cpu', '').isdigit():
                    socket_dict[0].append(int(element.replace('cpu', '')))
            socket_dict[0] = sorted(socket_dict[0])

        self.sockets_cores = socket_dict
        for numa in self.sockets_cores.keys():
            self.all_cores += self.sockets_cores[numa]

        self.sibling_cores = {}
        self.physical_cores = {}
        for numa in self.sockets_cores.keys():
            if numa not in self.sibling_cores.keys():
                self.sibling_cores[numa] = {}
                self.physical_cores[numa] = []
                self.offline_cores[numa] = []
                for core in self.sockets_cores[numa]:
                    err_message = "Unable to collect CORE ID info. \
                                  Core %s might be offline." % core
                    (rc, output) = self.command.run_command_debug_when_fail(
                        AccessLinux.GET_CORE_ID_CMD % (numa, core),
                        err_message)
                    if rc:
                        self.offline_cores[numa].append(core)
                        continue
                    core_id = output.replace('\n', '')
                    if core_id in self.sibling_cores[numa].keys():
                        self.sibling_cores[numa][core_id].append(core)
                    else:
                        self.sibling_cores[numa][core_id] = [core]
                        self.physical_cores[numa].append(core)
                self.sockets_cores[numa] = \
                    [c for c in self.sockets_cores[numa]
                        if c not in self.offline_cores[numa]]

        (rc, output) = self.command.run_command_warn_when_fail(
            AccessLinux.GET_WATCHDOG_CMD,
            "Unable to collect watchdog info.")
        if not rc and int(output) != 0:
            self.watchdog = Status.ACTIVE

        (rc, output) = self.command.run_command_warn_when_fail(
            AccessLinux.GET_NMI_WATCHDOG_CMD,
            "Unable to collect NMI watchdog info.")
        if not rc and int(output) != 0:
            self.nmi_watchdog = Status.ACTIVE

        (rc, output) = self.command.run_command_warn_when_fail(
            AccessLinux.GET_VM_STAT_INTERVAL_CMD,
            "Unable to collect VM stat interval info.")
        if not rc and int(output) > 0:
            self.vm_stat_interval = int(output)

    def collect_ppc_cpu_info(self, raw_cpuinfo):
        """ Collects CPU information for PPC systems
        """
        raw_cpuinfo = raw_cpuinfo.split('\n')
        total_cores = 0
        for line in raw_cpuinfo:
            if "processor" in line:
                total_cores += 1
        self.total_cores = total_cores
        for line in raw_cpuinfo:
            if "clock" in line:
                result = line.split(":")[1].split("MHz")[0].strip()
                self.actual_frequency = str(float(result)) + " MHz"
                break

        (rc, lscpu_output) = self.command.run_command_warn_when_fail(
            AccessLinux.GET_LSCPU_CMD,
            "Unable to collect cpu info.")
        for line in lscpu_output.split('\n'):
            if "Architecture" in line:
                architecture_str = line.split()[1].strip().lower()
                ibm_archs = Architecture.MODEL_TO_ARCH[Vendor.IBM]
                if architecture_str in \
                        ibm_archs.keys():
                    self.architecture = \
                        ibm_archs[architecture_str]
            if "Thread(s)" in line:
                threads_per_core = int(line.split(":")[1].strip())
                self.physical_cores_num = \
                    int(self.total_cores / threads_per_core)
                break

    def collect_arm_cpu_info(self, raw_cpuinfo):
        """ Collects CPU information for ARM systems
        """
        raw_cpuinfo = raw_cpuinfo.split('\n')
        total_cores = 0
        processor_pattern = re.compile("processor( *): \d+")
        for line in raw_cpuinfo:
            match = processor_pattern.search(line)
            if match:
                total_cores += 1

        (rc, lscpu_output) = self.command.run_command_warn_when_fail(
            AccessLinux.GET_LSCPU_CMD,
            "Unable to collect cpu info.")
        for line in lscpu_output.split('\n'):
            if "Architecture" in line:
                architecture_str = line.split()[1].strip().lower()
                arm_archs = Architecture.MODEL_TO_ARCH[Vendor.ARM]
                if architecture_str in arm_archs.keys():
                    self.architecture = arm_archs[architecture_str]
            if "Thread(s)" in line:
                threads_per_core = int(line.split(":")[1].strip())
                self.physical_cores_num = \
                    int(self.total_cores / threads_per_core)
                break

    def collect_intel_cpu_info(self, raw_cpuinfo, lscpu_output):
        """ Collects CPU information for Intel systems
        """
        raw_cpuinfo = raw_cpuinfo.split('\n')
        lscpu_output = lscpu_output.split('\n')
        for line in raw_cpuinfo:
            if "siblings" in line:
                self.total_cores = int(line.split(":")[1].strip())
                break
        self.physical_cores_num = 0
        for line in raw_cpuinfo:
            if "cpu cores" in line:
                self.physical_cores_num = int(line.split(":")[1].strip())
                break
        if not self.physical_cores_num:
            cpus = cpus_per_thread = 0
            for line in lscpu_output:
                if line.split()[0] == "CPU(s):":
                    cpus = int(line.split(":")[1].strip())
                elif "Thread(s) per core: " in line:
                    cpus_per_thread = int(line.split(":")[1].strip())
            if cpus and cpus_per_thread:
                self.physical_cores_num = cpus / cpus_per_thread
                self.total_cores = cpus
        for line in raw_cpuinfo:
            if "model name" in line:
                self.model = line.split(":")[1].strip()
                break
        for line in raw_cpuinfo:
            if "model name" not in line and "model" in line:
                cpu_model_number = line.split(":")[1].strip()
                break
        for line in raw_cpuinfo:
            if "cpu MHz" in line:
                self.actual_frequency = \
                    str(float(line.split(":")[1].strip())) + " MHz"
                break
        for line in raw_cpuinfo:
            if "cpu family" in line:
                cpu_family_number = line.split(":")[1].strip()
                break
        self.architecture = self.convert_architecture_number_to_name(
            int(cpu_model_number), int(cpu_family_number))
        (rc, output) = self.command.run_command_warn_when_fail(
            "%s %s" % (AccessLinux.DMIDECODE, "-s processor-frequency"),
            "Unable to collect processor frequency")
        if rc:
            self.max_frequency = None
        else:
            max_frequency_pattern = re.compile("\d+ MHz")
            for line in output.split("\n"):
                match = max_frequency_pattern.search(line)
                if match:
                    self.max_frequency = match.group(0)
                    break

    def convert_architecture_number_to_name(self,
                                            model_number, family_number):
        """ Converts model and family number to architecture name
        """
        if family_number == VendorConstants.INTEL_FAMILY:
            intel_archs = Architecture.MODEL_TO_ARCH[Vendor.INTEL]
            if model_number in \
                    intel_archs[VendorConstants.INTEL_FAMILY].keys():
                return intel_archs[VendorConstants.INTEL_FAMILY][model_number]

        return Architecture.UNKNOWN
