# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# @author: Simon Raviv
# @date: May 25, 2017

from mcra import MCRACounters
from performance.entities.unit import Unit
from performance.entities.counter import Counter
from performance.common.units import Units
from performance.common.units_measure import UnitsMeasure
from performance.common.device_ids import DeviceIDs
from performance.common.counters_names import CountersNames as Counters
from performance.common.utilization_reference import UtilizationReference \
        as Reference


class BlueFieldMCRACountres(MCRACounters):
    """ Represents BlueField counters definitions.
    """

    def __init__(self, device):
        super(BlueFieldMCRACountres, self).__init__(device=device)

        self._pci_link_gen_address = 0x137064

        self.units.append(Unit(Units.TPT, 0x0a6620))
        self.units.append(Unit(Units.PXDP, 0x0126420))
        self.units.append(Unit(Units.PXTL, 0x10f2a0))
        self.units.append(Unit(Units.PXTH, 0x18f720))
        self.units.append(Unit(Units.RXB, 0x0dc420))
        self.units.append(Unit(Units.RXT, 0x05a220))
        self.units.append(Unit(Units.RXW, 0x0650a0))
        self.units.append(Unit(Units.RXS, 0x0749a0))
        self.units.append(Unit(Units.RXC, 0x086a20))
        self.units.append(Unit(Units.SXP, 0x0389a0))
        self.units.append(Unit(Units.SXP_E2E_CACHE,
                               0x039650, 8, 0x039654, 0x039640, 8))
        self.units.append(Unit(Units.SXW, 0x02cd20))
        self.units.append(Unit(Units.RXPS, 0x043020))
        self.units.append(Unit(Units.RXPS_E2E_CACHE,
                               0x040050, 8, 0x040054, 0x040040, 8))

        # TPT #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=258,
                             name=Counters.TPT_L0_MTT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=257,
                             name=Counters.TPT_L0_MTT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=263,
                             name=Counters.TPT_L1_MTT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=262,
                             name=Counters.TPT_L1_MTT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=5,
                             utilization_bad=15))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=273,
                             name=Counters.TPT_L0_MPT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=272,
                             name=Counters.TPT_L0_MPT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=278,
                             name=Counters.TPT_L1_MPT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=277,
                             name=Counters.TPT_L1_MPT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=5,
                             utilization_bad=15))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=270,
                             name=Counters.TPT_INDIRECT_MEM_KEY,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=1,
                             utilization_bad=1))
        # PXTH #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXTH),
                             selector=367,
                             name=Counters.PXT_PERF_RD_ICMC_PUSH_LINK0,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=1,
                             utilization_bad=5))
        # PXTL #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXTL),
                             selector=343,
                             name=Counters.PXT_PXD_READY_BP,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=1,
                             utilization_bad=5))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXTL),
                             selector=346,
                             name=Counters.PXT_PCI_READ_BP,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXTL),
                             selector=345,
                             name=Counters.PXT_PCI_WRITE_BP,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXTL),
                             selector=325,
                             name=Counters.
                             PXT_PCI_READ_STUCK_DUE_TO_NO_READ_ENGINES,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXTL),
                             selector=326,
                             name=Counters.
                             PXT_PCI_READ_STUCK_DUE_TO_BYTE_LIMIT,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXTL),
                             selector=330,
                             name=Counters.PXT_PCI_READ_STUCK_DUE_TO_ORDERING,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        # RXB #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=9,
                             name=Counters.RXB_BW_COUNT_PERF_COUNT_1_1,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=10,
                             name=Counters.RXB_BW_COUNT_PERF_COUNT_1_2,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=12,
                             name=Counters.RXB_BW_COUNT_PERF_COUNT_0_1,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=13,
                             name=Counters.RXB_BW_COUNT_PERF_COUNT_0_2,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=52,
                             name=Counters.RXB_RXS_NO_SLOW_PATH_CREDITS,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=48,
                             name=Counters.
                             RXB_RXT_NO_SLOW_PATH_CRED_PERF_COUNT,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=17,
                             name=Counters.
                             RXB_PORT0_BUFFER_FULL_PERF_COUNT_PORT0,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=15,
                             name=Counters.
                             RXB_PORT1_BUFFER_FULL_PERF_COUNT_PORT1,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=1,
                             utilization_bad=10))
        # RXPS #
        # These two must be in this order. When setting a to 0,
        # even counters count wraparound Odd counters count clocks.
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=0,
                             name=Counters.DEVICE_CLOCKS_WRAPAROUND,
                             units=UnitsMeasure.CYCLES))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=0,
                             name=Counters.DEVICE_CLOCKS,
                             units=UnitsMeasure.CYCLES))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=26,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING0_RSE_WORK_RATE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=45,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING1_RSE_WORK_RATE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=30,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING0_CACHE_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=29,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING0_CACHE_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=49,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING1_CACHE_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=48,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING1_CACHE_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=28,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING0_CACHE_TOT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=47,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING1_CACHE_TOT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS_E2E_CACHE),
                             selector=15,
                             name=Counters.RXPS_E2E_CACHE_TOTAL_LOOKUPS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS_E2E_CACHE),
                             selector=24,
                             name=Counters.RXPS_E2E_CACHE_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS_E2E_CACHE),
                             selector=33,
                             name=Counters.RXPS_E2E_CACHE_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS,
                             utilization_good=5,
                             utilization_bad=15))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS_E2E_CACHE),
                             selector=42,
                             name=Counters.RXPS_E2E_CACHE_LEARN,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS,
                             utilization_good=1,
                             utilization_bad=1))
        # RXW #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXW),
                             selector=70,
                             name=Counters.RXW_PERF_COUNT_TPT_CREDIT,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXW),
                             selector=66,
                             name=Counters.RXW_PERF_WB_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXW),
                             selector=65,
                             name=Counters.RXW_PERF_WB_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        # RXS #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXS),
                             selector=23,
                             name=Counters.RXS_NO_PXT_CREDITS,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        # RXT #
        self.counters.append(Counter(unit=self.get_unit(Units.RXT),
                             selector=21,
                             name=Counters.RXT_CTRL_PERF_SLICE_LOAD_SLOW,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(unit=self.get_unit(Units.RXT),
                             selector=20,
                             name=Counters.RXT_CTRL_PERF_SLICE_LOAD_FAST,
                             units=UnitsMeasure.PACKETS))
        # PXDP #
        if self._check_pci_integrity_issue() is False:
            self.counters.append(Counter(
                                 unit=self.get_unit(Units.PXDP),
                                 selector=723,
                                 name=Counters.PXDP_RX_128B_TOTAL))
            self.counters.append(Counter(
                                 unit=self.get_unit(Units.PXDP),
                                 selector=724,
                                 name=Counters.PXDP_RX_128B_DATA))
            self.counters.append(Counter(
                                 unit=self.get_unit(Units.PXDP),
                                 selector=650,
                                 name=Counters.PXDP_TX_128B_DATA))
        # RXC #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=104,
                             name=Counters.RXC_EQ_ALL_SLICES_BUSY,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=180,
                             name=Counters.RXC_CQ_ALL_SLICES_BUSY,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=43,
                             name=Counters.RXC_MSIX_ALL_SLICES_BUSY,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=227,
                             name=Counters.RXC_CQE_ZIP_OPEN_SESSION,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS,
                             utilization_good=15,
                             utilization_bad=50))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=228,
                             name=Counters.RXC_CQE_ZIP_MERGING_CQE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=230,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_EQE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=231,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_TIMEOUT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=232,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_NOT_MATCH,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=233,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_PX_IDLE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=234,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_S2CQE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=229,
                             name=Counters.RXC_CQE_ZIP_WRITING_8_CQES,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=235,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_LRO,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        # SXW #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=161,
                             name=Counters.SXW_QP_DONE_DUE_TO_LIMITED,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=158,
                             name=Counters.SXW_QP_DONE_DUE_TO_VL_LIMITED,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=159,
                             name=Counters.SXW_QP_DONE_DUE_TO_DESCHED,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=160,
                             name=Counters.SXW_QP_DONE_DUE_TO_WORK_DONE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=10,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=162,
                             name=Counters.SXW_QP_DONE_DUE_TO_E2E_CREDITS,
                             units=UnitsMeasure.EVENTS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=242,
                             name=Counters.SXW_PACKET_SEND_SXW2SXP_GO_VID,
                             units=UnitsMeasure.PACKETS))
        # SXP #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=139,
                             name=Counters.SXP_LINE_TRANSMITTED_PORT0,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=140,
                             name=Counters.SXP_LINE_TRANSMITTED_PORT1,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=141,
                             name=Counters.SXP_LINE_TRANSMITTED_LOOPBACK,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=29,
                             name=Counters.
                             SXP_STEERING_PERF_COUNT_STEERING0_RSE_WORK_RATE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=27,
                             name=Counters.
                             SXP_STEERING_PERF_COUNT_STEERING1_RSE_WORK_RATE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=14,
                             name=Counters.SXP_PERF_COUNT_STEERING0_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=16,
                             name=Counters.SXP_PERF_COUNT_STEERING0_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=13,
                             name=Counters.SXP_PERF_COUNT_STEERING1_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=15,
                             name=Counters.SXP_PERF_COUNT_STEERING1_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=23,
                             name=Counters.
                             SXP_PERF_COUNT_STEERING0_CACHE_1_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=18,
                             name=Counters.
                             SXP_PERF_COUNT_STEERING1_CACHE_1_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=22,
                             name=Counters.SXP_STEERING_0_TOTAL_CACHE_ACCESS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=17,
                             name=Counters.SXP_STEERING_1_TOTAL_CACHE_ACCESS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP_E2E_CACHE),
                             selector=15,
                             name=Counters.SXP_E2E_CACHE_TOTAL_LOOKUPS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP_E2E_CACHE),
                             selector=24,
                             name=Counters.SXP_E2E_CACHE_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP_E2E_CACHE),
                             selector=33,
                             name=Counters.SXP_E2E_CACHE_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=5,
                             utilization_bad=15))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP_E2E_CACHE),
                             selector=42,
                             name=Counters.SXP_E2E_CACHE_LEARN,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=1))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=182,
                             name=Counters.SXP_BW_COUNT_PERF_COUNT_0_1))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=183,
                             name=Counters.SXP_BW_COUNT_PERF_COUNT_0_2))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=179,
                             name=Counters.SXP_BW_COUNT_PERF_COUNT_1_1))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=180,
                             name=Counters.SXP_BW_COUNT_PERF_COUNT_1_2))

    @staticmethod
    def is_device_id_ok(id):
        if id in DeviceIDs.BLUEFIELD:
            return True
        return False

    @staticmethod
    def set_latency_counters(mf):
        """ Set latency counters for sampling.
        """
        # Enable PXTL latency measurements:
        mf.write_field(val=1, addr=0x10d500, startBit=1, size=1)

        # Set flavor to measuring PXTL to response:
        mf.write_field(val=1, addr=0x10d508, startBit=0, size=1)
        mf.write_field(val=1, addr=0x10d508, startBit=1, size=1)

        # PXTL to response:
        # Reset the counters to NULL:
        mf.write_field(val=1, addr=0x10d900, startBit=0, size=1)
        mf.write_field(val=1, addr=0x10d900, startBit=1, size=1)
        mf.write_field(val=1, addr=0x10d900, startBit=2, size=1)

        # Enables the counter that will be used to measure latency:
        mf.write_field(val=1, addr=0x10d904, startBit=0, size=1)

        # Release counter lock:
        mf.write_field(val=0, addr=0x10d904, startBit=1, size=1)

        # Set flavor to measuring PXTH to response:
        mf.write_field(val=0, addr=0x18511c, startBit=0, size=1)

        # Enable PXTH to response latency measurements:
        mf.write_field(val=1, addr=0x198008, startBit=1, size=1)

        # PXTH to response:
        # Reset the counters to NULL:
        mf.write_field(val=1, addr=0x198080, startBit=0, size=1)
        mf.write_field(val=1, addr=0x198080, startBit=1, size=1)
        mf.write_field(val=1, addr=0x198080, startBit=2, size=1)

        # Enables the counter that will be used to measure latency:
        mf.write_field(val=1, addr=0x198084, startBit=0, size=1)

        # Release counter lock:
        mf.write_field(val=0, addr=0x198084, startBit=1, size=1)

    def get_latency_counters(self, mf, dev_clock):
        """ Returns latency counters.
        """
        # Reset the counters to NULL:
        mf.write_field(val=0, addr=0x10d900, startBit=0, size=1)
        mf.write_field(val=0, addr=0x10d900, startBit=1, size=1)
        mf.write_field(val=0, addr=0x10d900, startBit=2, size=1)
        mf.write_field(val=1, addr=0x198080, startBit=0, size=1)
        mf.write_field(val=1, addr=0x198080, startBit=1, size=1)
        mf.write_field(val=1, addr=0x198080, startBit=2, size=1)

        # Lock counters for sum/cnt:
        mf.write_field(val=0, addr=0x10d904, startBit=0, size=1)
        mf.write_field(val=1, addr=0x10d904, startBit=1, size=1)
        mf.write_field(val=0, addr=0x198084, startBit=0, size=1)
        mf.write_field(val=1, addr=0x198084, startBit=1, size=1)

        # Read PXTH latency counters:
        pxth_to_pcie_response_latency_sum_31_0 = mf.read4(0x19809c)
        pxth_to_pcie_response_latency_sum_63_32 = mf.read4(0x198098)
        pxth_to_pcie_response_latency_cnt_31_0 = mf.read4(0x198094)
        pxth_to_pcie_response_min_latency = mf.read4(0x198088)
        pxth_to_pcie_response_max_latency = mf.read4(0x19808c)

        # Read PXTL latency counters:
        #pxtl_to_pcie_response_latency_sum_31_0 = mf.read4(0x10d91c)
        #pxtl_to_pcie_response_latency_sum_63_32 = mf.read4(0x10d918)
        #pxtl_to_pcie_response_latency_cnt_31_0 = mf.read4(0x10d914)
        #pxtl_to_pcie_response_min_latency = mf.read4(0x10d908)
        #pxtl_to_pcie_response_max_latency = mf.read4(0x10d90c)

        pxth_to_pcie_response_latency = 0
        if pxth_to_pcie_response_latency_cnt_31_0:
            pxth_to_pcie_response_latency_sum_total = \
                pxth_to_pcie_response_latency_sum_31_0 + \
                (pxth_to_pcie_response_latency_sum_63_32 << 32)
            pxth_to_pcie_response_latency = \
                pxth_to_pcie_response_latency_sum_total * 1000 / \
                pxth_to_pcie_response_latency_cnt_31_0 / dev_clock
        pxth_to_pcie_response_latency_min = \
            pxth_to_pcie_response_min_latency * 1000 / dev_clock
        pxth_to_pcie_response_latency_max = \
            pxth_to_pcie_response_max_latency * 1000 / dev_clock

        #pxtl_to_pcie_response_latency = 0
        #if pxtl_to_pcie_response_latency_cnt_31_0:
        #    pxtl_to_pcie_response_latency_sum_total = \
        #        pxtl_to_pcie_response_latency_sum_31_0 + \
        #        (pxtl_to_pcie_response_latency_sum_63_32 << 32)
        #    pxtl_to_pcie_response_latency = \
        #        pxtl_to_pcie_response_latency_sum_total * 1000 / \
        #        pxtl_to_pcie_response_latency_cnt_31_0 / dev_clock
        #pxtl_to_pcie_response_latency_min = \
        #    pxtl_to_pcie_response_min_latency * 1000 / dev_clock
        #pxtl_to_pcie_response_latency_max = \
        #    pxtl_to_pcie_response_max_latency * 1000 / dev_clock

        # In case of counter overflow:
        if pxth_to_pcie_response_latency == 0:
            pxth_to_pcie_response_latency_min = 0
            pxth_to_pcie_response_latency_max = 0

        #if pxtl_to_pcie_response_latency == 0:
        #    pxtl_to_pcie_response_latency_min = 0
        #    pxtl_to_pcie_response_latency_max = 0

        # Initialize counters objects:
        self.regular_counters.append(
            Counter(name=Counters.PCI_LATENCY_LINK0_AVG,
                    value=0))#pxtl_to_pcie_latency))
        self.regular_counters.append(
            Counter(name=Counters.PCI_LATENCY_LINK0_MIN,
                    value=0))#pxtl_to_pcie_latency_min))
        self.regular_counters.append(
            Counter(name=Counters.PCI_LATENCY_LINK0_MAX,
                    value=0))#pxtl_to_pcie_latency_max))
        self.regular_counters.append(
            Counter(name=Counters.PXT_TO_ARM_LATENCY_AVG,
                    value=(pxth_to_pcie_response_latency)))
        self.regular_counters.append(
            Counter(name=Counters.PXT_TO_ARM_LATENCY_MIN,
                    value=(pxth_to_pcie_response_latency_min)))
        self.regular_counters.append(
            Counter(name=Counters.PXT_TO_ARM_LATENCY_MAX,
                    value=(pxth_to_pcie_response_latency_max)))

    def _check_pci_integrity_issue(self):
        """ Check FW PCI integrity issues.
            This method intendeds to guard from FW PCI
            and the performance plugin to use the same counters
            in PCI units, in which the FW rely on in some scenarios.
            The issue will occur based on the condition below.
        """
        if self._get_pci_link_gen() == 4:
            return True
        else:
            return False
