# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Simon Raviv
# @date: December 05, 2017


from performance.common.counters_names import CountersNames as Counters


class CountersNamesDescriptionMap(object):
    """ Describes names mapping between counter's name to it's description.
    """

    def __init__(self):
        self.__names_map = dict()
        self.__initialize_map()

    def __initialize_tpt(self):
        """ Initialize TPT unit descriptions.
        """
        self.__names_map[Counters.TPT_L0_MTT_MISS] = \
            ("Address translation level 0 cache miss")
        self.__names_map[Counters.TPT_L0_MTT_HIT] = \
            ("Address translation level 0 cache hit")
        self.__names_map[Counters.TPT_L1_MTT_MISS] = \
            ("Address translation level 1 cache miss")
        self.__names_map[Counters.TPT_L1_MTT_HIT] = \
            ("Address translation level 1 cache hit")
        self.__names_map[Counters.TPT_L0_MPT_MISS] = \
            ("Memory region context level 0 cache miss")
        self.__names_map[Counters.TPT_L0_MPT_HIT] = \
            ("Memory region context level 0 cache hit")
        self.__names_map[Counters.TPT_L1_MPT_MISS] = \
            ("Memory region context level 1 cache miss")
        self.__names_map[Counters.TPT_L1_MPT_HIT] = \
            ("Memory region context level 1 cache hit")
        self.__names_map[Counters.TPT_INDIRECT_MEM_KEY] = \
            ("Indirect memory key used in address translation")

    def __initialize_rxb(self):
        """ Initialize RXB unit descriptions.
        """
        self.__names_map[Counters.RXB_RXS_NO_SLOW_PATH_CREDITS] = \
            ("Number of cycles RX packet buffer is blocked"
             " by packet scatter unit")
        self.__names_map[Counters.RXB_RXT_NO_SLOW_PATH_CRED_PERF_COUNT] = \
            ("Number of cycles RX packet buffer is blocked by"
             " packet processing unit")
        self.__names_map[Counters.RXB_PORT0_BUFFER_FULL_PERF_COUNT_PORT0] = \
            ("Number of cycles that RX packet buffer of port 1 was full")
        self.__names_map[Counters.RXB_PORT1_BUFFER_FULL_PERF_COUNT_PORT1] = \
            ("Number of cycles that RX packet buffer of port 2 was full")
        self.__names_map[Counters.RXB_LRO_FIFO_PERF_COUNT2] = \
            ("Number of Large Received Offload sessions that"
             " were closed due to timeout")
        self.__names_map[Counters.RXB_BW_COUNT_PERF_COUNT_0_2] = \
            ("Number of RX packets")
        self.__names_map[Counters.RXB_S1HOST_ARBITER_PERF_DESC_BUFFER_CREDIT_HOST0] = \
            ("Number of cycles that RX descriptor handling is blocked by RX Buffer")
        self.__names_map[Counters.RXB_RXT_S1PSA_SWITCH_PERF_COUNT] = \
            ("Number of cycles that RX descriptor handling is blocked by RX Translation unit")
        self.__names_map[Counters.RXB_S1PSA_SWITCH_PERF_COUNT] = \
            ("Number of cycles that RX descriptor handling is blocked by PSA")
        self.__names_map[Counters.RXB_RXD_S1PSA_SWITCH_PERF_COUNT] = \
            ("Number of cycles that RX descriptor handling is blocked by RXD")

    def __initialize_ptb(self):
        """ Initialize PTB unit descriptions.
        """
        self.__names_map[Counters.PTB_PORT_IF_48B_PORT_0] = \
            ("Number of cycles that RX packet buffer of port 1 was full")
        self.__names_map[Counters.PTB_PORT_IF_48B_PORT_1] = \
            ("Number of cycles that RX packet buffer of port 2 was full")
        self.__names_map[Counters.PTB_PORT_IF_PACKETS_ALL_PORTS] = \
            ("Number of RX packets")

    def __initialize_rxt(self):
        """ Initialize RXT unit descriptions.
        """
        self.__names_map[Counters.RXT_CTRL_PERF_SLICE_LOAD_SLOW] = \
            ("Number of RX packets that went through steering")
        self.__names_map[Counters.RXT_CTRL_PERF_SLICE_LOAD_FAST] = \
            ("Number of RX packets that went through steering on fast path")
        self.__names_map[
            Counters.RXT_STEERING_PERF_COUNT_STEERING0_RSE_WORK_RATE] = \
            ("Number of RX steering events on pipe 0")
        self.__names_map[
            Counters.RXT_STEERING_PERF_COUNT_STEERING1_RSE_WORK_RATE] = \
            ("Number of RX steering events on pipe 1")
        self.__names_map[
            Counters.RXT_STEERING_PERF_COUNT_STEERING0_CACHE_HIT] = \
            ("Number of RX steering rules cache hit on pipe 0")
        self.__names_map[
            Counters.RXT_STEERING_PERF_COUNT_STEERING1_CACHE_HIT] = \
            ("Number of RX steering rules cache hit on pipe 1")
        self.__names_map[
            Counters.RXT_STEERING_PERF_COUNT_STEERING0_CACHE_MISS] = \
            ("Number of RX steering rules cache miss on pipe 0")
        self.__names_map[
            Counters.RXT_STEERING_PERF_COUNT_STEERING1_CACHE_MISS] = \
            ("Number of RX steering rules cache miss on pipe 1")
        self.__names_map[
            Counters.RXT_STEERING_PERF_COUNT_STEERING0_CACHE_TOT] = \
            ("Number of RX steering rules cache total access on pipe 0")
        self.__names_map[
            Counters.RXT_STEERING_PERF_COUNT_STEERING1_CACHE_TOT] = \
            ("Number of RX steering rules cache total access on pipe 1")

    def __initialize_rxw(self):
        """ Initialize RXW unit description.
        """
        self.__names_map[Counters.RXW_PERF_COUNT_TPT_CREDIT] = \
            ("Number of cycles that RX descriptor handling is blocked by"
             " internal MMU")
        self.__names_map[Counters.RXW_PERF_WB_MISS] = \
            ("Receive Work Queue Element cache miss")
        self.__names_map[Counters.RXW_PERF_WB_HIT] = \
            ("Receive Work Queue Element cache hit")

    def __initialize_rxs(self):
        """ Initialize RXS unit description.
        """
        self.__names_map[Counters.RXS_NO_PXT_CREDITS] = \
            ("Number of cycles packet scatter unit is"
             " blocked by PCIe unit")

    def __initialize_rxc(self):
        """ Initialize RXC unit description.
        """
        self.__names_map[Counters.RXC_EQ_ALL_SLICES_BUSY] = \
            ("Number of cycles that all state machines"
             " for event queues are busy")
        self.__names_map[Counters.RXC_CQ_ALL_SLICES_BUSY] = \
            ("Number of cycles that all state machines"
             " for completion queues are busy")
        self.__names_map[Counters.RXC_MSIX_ALL_SLICES_BUSY] = \
            ("Number of cycles that all state machines"
             " for MSI-X are busy")
        self.__names_map[Counters.RXC_CQE_ZIP_OPEN_SESSION] = \
            ("Number of compression sessions for"
             " completion queue entries")
        self.__names_map[Counters.RXC_CQE_ZIP_MERGING_CQE] = \
            ("Number of compressed Completion Queue Entries")
        self.__names_map[Counters.RXC_CQE_ZIP_WRITING_8_CQES] = \
            ("Number of strides with 8 compressed Completion"
             " Queue Entries")
        self.__names_map[Counters.RXC_CLOSING_ZIP_SESSION_EQE] = \
            ("Number of compression sessions that were closed"
             " due to Event Queue Entry session")
        self.__names_map[Counters.RXC_CLOSING_ZIP_SESSION_TIMEOUT] = \
            ("Number of compression sessions that were closed"
             " due to timeout")
        self.__names_map[Counters.RXC_CLOSING_ZIP_SESSION_NOT_MATCH] = \
            ("Number of compression sessions that were closed"
             " due to mismatch")
        self.__names_map[Counters.RXC_CLOSING_ZIP_SESSION_PX_IDLE] = \
            ("Number of compression sessions that were closed"
             " due to PCIe idle condition state")
        self.__names_map[Counters.RXC_CLOSING_ZIP_SESSION_S2CQE] = \
            ("Number of compression sessions that were closed"
             " due to scatter packet to Completion Queue Entry")
        self.__names_map[Counters.RXC_CLOSING_ZIP_SESSION_LRO] = \
            ("Number of compression sessions that were closed"
             " due to Large Receive Offload session")

    def __initialize_rxd(self):
        """ Initialize RXD unit description.
        """
        self.__names_map[Counters.RXD_CTRL_IPSEC_PACKET] = \
            ("Number of IPsec packets")
        self.__names_map[Counters.RXD_CTRL_G_CREDIT_PSA1_SLICE_HOST_0] = \
            ("Number of cycles that RX descriptor handling is blocked on PSA by RXD")

    def __initialize_sxw(self):
        """ Initialize SXW unit description.
        """
        self.__names_map[Counters.SXW_QP_DONE_DUE_TO_VL_LIMITED] = \
            ("Number of times TX descriptor handling stopped due to back"
             " pressure from virtual lane")
        self.__names_map[Counters.SXW_QP_DONE_DUE_TO_DESCHED] = \
            ("Number of times TX descriptor handling"
             " stopped due to de-schedule")
        self.__names_map[Counters.SXW_QP_DONE_DUE_TO_WORK_DONE] = \
            ("Number of times TX descriptor handling"
             " stopped due to no more work")
        self.__names_map[Counters.SXW_QP_DONE_DUE_TO_LIMITED] = \
            ("Number of times TX descriptor handling"
             " stopped due to limited state")
        self.__names_map[Counters.SXW_QP_DONE_DUE_TO_E2E_CREDITS] = \
            ("Number of times TX descriptor handling stopped due to end"
             " to end credits")
        self.__names_map[Counters.SXW_PACKET_SEND_SXW2SXP_GO_VID] = \
            ("Number of TX packets")
        self.__names_map[Counters.SXW_PERF_COUNT_STEERING_HIT] = \
            ("Number of TX steering hits")
        self.__names_map[Counters.SXW_PERF_COUNT_STEERING_MISS] = \
            ("Number of TX steering misses")
        self.__names_map[Counters.SXW_STEERING_PERF_COUNT_CACHE_HIT] = \
            ("Number of TX steering rules cache hits")
        self.__names_map[Counters.SXW_STEERING_PERF_COUNT_CACHE_MISS] = \
            ("Number of TX steering rules cache misses")
        self.__names_map[Counters.SXW_STEERING_PERF_COUNT_CACHE_TOT] = \
            ("Number of TX steering rules cache accesses")
        self.__names_map[Counters.SXW_PERF_COUNT_STEERING_RSE0] = \
            ("Number of TX steering events on the pipe")

    def __initialize_sxd(self):
        """ Initialize SXD unit description.
        """
        self.__names_map[Counters.SXD_NO_SLOW_PATH_CREDITS] = \
            ("Number of cycles scheduling & QoS unit is blocked"
             " by TX descriptor handling")

    def __initialize_rxps(self):
        """ Initialize RXPS unit description.
        """
        self.__names_map[Counters.RXPS_E2E_CACHE_TOTAL_LOOKUPS] = \
            ("Number of RX steering learning cache total lookups")
        self.__names_map[Counters.RXPS_E2E_CACHE_HIT] = \
            ("Number of RX steering learning cache hit")
        self.__names_map[Counters.RXPS_E2E_CACHE_MISS] = \
            ("Number of RX steering learning cache miss")
        self.__names_map[Counters.RXPS_E2E_CACHE_LEARN] = \
            ("Number of RX steering learning cache learned rule")
        self.__names_map[Counters.DEVICE_CLOCKS] = \
            ("The frequency of the chip")
        self.__names_map[Counters.DEVICE_CLOCKS_WRAPAROUND] = \
            ("The wraparound of the chip frequency counter")
        self.__names_map[
            Counters.RXPS_STEERING_PERF_COUNT_STEERING0_RSE_WORK_RATE] = \
            ("Number of RX steering events on pipe 0")
        self.__names_map[
            Counters.RXPS_STEERING_PERF_COUNT_STEERING1_RSE_WORK_RATE] = \
            ("Number of RX steering events on pipe 1")
        self.__names_map[
            Counters.RXPS_STEERING_PERF_COUNT_STEERING0_CACHE_HIT] = \
            ("Number of RX steering rules cache hit on pipe 0")
        self.__names_map[
            Counters.RXPS_STEERING_PERF_COUNT_STEERING0_CACHE_MISS] = \
            ("Number of RX steering rules cache miss on pipe 0")
        self.__names_map[
            Counters.RXPS_STEERING_PERF_COUNT_STEERING1_CACHE_HIT] = \
            ("Number of RX steering rules cache hit on pipe 1")
        self.__names_map[
            Counters.RXPS_STEERING_PERF_COUNT_STEERING1_CACHE_MISS] = \
            ("Number of RX steering rules cache miss on pipe 1")
        self.__names_map[
            Counters.RXPS_STEERING_PERF_COUNT_STEERING0_CACHE_TOT] = \
            ("Number of RX steering rules cache total access on pipe 0")
        self.__names_map[
            Counters.RXPS_STEERING_PERF_COUNT_STEERING1_CACHE_TOT] = \
            ("Number of RX steering rules cache total access on pipe 1")

    def __initialize_pxt(self):
        """ Initialize PXT unit description.
        """
        self.__names_map[Counters.PXT_PXD_READY_BP] = \
            ("Number of cycles PCIe unit is blocked")
        self.__names_map[Counters.PXT_PCI_WRITE_BP] = \
            ("Number of cycles that the PCIe unit had"
             " outbound posted write requests but could not perform"
             " the operation due to insufficient posted credits")
        self.__names_map[Counters.PXT_PCI_READ_BP] = \
            ("Number of cycles that the PCIe unit had outbound"
             " non-posted read requests but could not perform"
             " the operation due to insufficient"
             " non-posted credits")
        self.__names_map[
            Counters.PXT_PCI_READ_STUCK_DUE_TO_NO_COMPLETION_BUFFER] = \
            ("Number of cycles that the PCIe unit had outbound"
             " read requests but could not perform the operation due"
             " to insufficient available completion buffers")
        self.__names_map[
            Counters.PXT_PCI_READ_STUCK_DUE_TO_NO_READ_ENGINES] = \
            ("Number of cycles that the PCIe unit had outbound"
             " read requests but could not perform the operation due"
             " to insufficient read engines")
        self.__names_map[Counters.PXT_PCI_READ_STUCK_DUE_TO_BYTE_LIMIT] = \
            ("Number of cycles that the PCIe unit had outbound"
             " read requests but could not perform the operation due"
             " to insufficient available completion buffers")
        self.__names_map[Counters.PXT_PCI_READ_STUCK_DUE_TO_ORDERING] = \
            ("Number of cycles that the PCIe unit had outbound"
             " read requests but could not perform the operation"
             " due to PCIe ordering rules")
        self.__names_map[Counters.PXT_PXD_READY_BP] = \
            ("Number of cycles PCIe unit is blocked")
        self.__names_map[Counters.PXT_PCI_WRITE_BP] = \
            ("Number of cycles that the PCIe unit had outbound"
             " posted write requests but could not perform the "
             "operation due to insufficient posted credits")
        self.__names_map[Counters.PXT_PCI_READ_BP] = \
            ("Number of cycles that the PCIe unit had outbound"
             " non-posted read requests but could not perform the "
             "operation due to insufficient non-posted credits")
        self.__names_map[
            Counters.PXT_PCI_READ_STUCK_DUE_TO_NO_COMPLETION_BUFFER] = \
            ("Number of cycles that the PCIe unit had outbound"
             " read requests but could not perform the operation"
             " due to insufficient available completion buffers")
        self.__names_map[Counters.PXT_LACK_IN_CMP_ENG_BP] = \
            ("Number of cycles that the PCIe unit had outbound"
             " read requests but could not perform the operation due"
             " to insufficient read engines")
        self.__names_map[Counters.PXT_PERF_RD_ICMC_PUSH_LINK0] = \
            ("General ICM cache miss - Read from host memory")

    def __initialize_sxp(self):
        """ Initialize SXP unit description.
        """
        self.__names_map[Counters.SXP_SXP_LINE_TRANSMITTED_PERF_COUNT0] = \
            ("Number of transmitted port lines")
        self.__names_map[Counters.SXP_LINE_TRANSMITTED_PORT0] = \
            ("Number of lines transmitted on port 1")
        self.__names_map[Counters.SXP_LINE_TRANSMITTED_PORT1] = \
            ("Number of lines transmitted on port 2")
        self.__names_map[Counters.SXP_LINE_TRANSMITTED_LOOPBACK] = \
            ("Number of loop back lines transmitted")
        self.__names_map[
            Counters.SXP_STEERING_PERF_COUNT_STEERING0_RSE_WORK_RATE] = \
            ("Number of TX steering events on pipe 0")
        self.__names_map[
            Counters.SXP_STEERING_PERF_COUNT_STEERING1_RSE_WORK_RATE] = \
            ("Number of TX steering events on pipe 1")
        self.__names_map[Counters.SXP_PERF_COUNT_STEERING0_HIT] = \
            ("Number of TX steering hits on pipe 0")
        self.__names_map[Counters.SXP_PERF_COUNT_STEERING0_MISS] = \
            ("Number of TX steering misses on pipe 0")
        self.__names_map[Counters.SXP_PERF_COUNT_STEERING1_HIT] = \
            ("Number of TX steering hits on pipe 1")
        self.__names_map[Counters.SXP_PERF_COUNT_STEERING1_MISS] = \
            ("Number of TX steering misses on pipe 1")
        self.__names_map[Counters.SXP_PERF_COUNT_STEERING0_CACHE_1_MISS] = \
            ("Number of TX steering rules cache misses on pipe 0")
        self.__names_map[Counters.SXP_PERF_COUNT_STEERING1_CACHE_1_MISS] = \
            ("Number of TX steering rules cache misses on pipe 1")
        self.__names_map[Counters.SXP_PERF_COUNT_STEERING0_CACHE_1_HIT] = \
            ("Number of TX steering rules cache hits on pipe 0")
        self.__names_map[Counters.SXP_PERF_COUNT_STEERING1_CACHE_1_HIT] = \
            ("Number of TX steering rules cache hits on pipe 1")
        self.__names_map[Counters.SXP_STEERING_0_TOTAL_CACHE_ACCESS] = \
            ("Number of TX steering rules cache total access on pipe 0")
        self.__names_map[Counters.SXP_STEERING_1_TOTAL_CACHE_ACCESS] = \
            ("Number of TX steering rules cache total access on pipe 1")
        self.__names_map[Counters.SXP_E2E_CACHE_TOTAL_LOOKUPS] = \
            ("Number of TX steering learning cache total lookups")
        self.__names_map[Counters.SXP_E2E_CACHE_HIT] = \
            ("Number of TX steering learning cache hit")
        self.__names_map[Counters.SXP_E2E_CACHE_MISS] = \
            ("Number of TX steering learning cache miss")
        self.__names_map[Counters.SXP_E2E_CACHE_LEARN] = \
            ("Number of TX steering learning cache learned rule")

    def __initialize_psa(self):
        """ Initialize PSA unit description.
        """
        self.__names_map[
            Counters.PSA_TX0_STEERING_PERF_COUNT_STEERING0_RSE_WORK_RATE] = \
            ("Number of TX PSA0 steering events on pipe 0")
        self.__names_map[
            Counters.PSA_TX0_STEERING_PERF_COUNT_STEERING1_RSE_WORK_RATE] = \
            ("Number of TX PSA0 steering events on pipe 1")
        self.__names_map[Counters.PSA_TX0_PERF_COUNT_STEERING0_CACHE_1_MISS] = \
            ("Number of TX PSA0 steering rules cache misses on pipe 0")
        self.__names_map[Counters.PSA_TX0_PERF_COUNT_STEERING1_CACHE_1_MISS] = \
            ("Number of TX PSA0 steering rules cache misses on pipe 1")
        self.__names_map[Counters.PSA_TX0_PERF_COUNT_STEERING0_CACHE_1_HIT] = \
            ("Number of TX PSA0 steering rules cache hits on pipe 0")
        self.__names_map[Counters.PSA_TX0_PERF_COUNT_STEERING1_CACHE_1_HIT] = \
            ("Number of TX PSA0 steering rules cache hits on pipe 1")
        self.__names_map[Counters.PSA_TX0_STEERING_0_TOTAL_CACHE_ACCESS] = \
            ("Number of TX PSA0 steering rules cache total access on pipe 0")
        self.__names_map[Counters.PSA_TX0_STEERING_1_TOTAL_CACHE_ACCESS] = \
            ("Number of TX PSA0 steering rules cache total access on pipe 1")
        self.__names_map[
            Counters.PSA_TX1_STEERING_PERF_COUNT_STEERING0_RSE_WORK_RATE] = \
            ("Number of TX PSA1 steering events on pipe 0")
        self.__names_map[
            Counters.PSA_TX1_STEERING_PERF_COUNT_STEERING1_RSE_WORK_RATE] = \
            ("Number of TX PSA1 steering events on pipe 1")
        self.__names_map[Counters.PSA_TX1_PERF_COUNT_STEERING0_CACHE_1_MISS] = \
            ("Number of TX PSA1 steering rules cache misses on pipe 0")
        self.__names_map[Counters.PSA_TX1_PERF_COUNT_STEERING1_CACHE_1_MISS] = \
            ("Number of TX PSA1 steering rules cache misses on pipe 1")
        self.__names_map[Counters.PSA_TX1_PERF_COUNT_STEERING0_CACHE_1_HIT] = \
            ("Number of TX PSA1 steering rules cache hits on pipe 0")
        self.__names_map[Counters.PSA_TX1_PERF_COUNT_STEERING1_CACHE_1_HIT] = \
            ("Number of TX PSA1 steering rules cache hits on pipe 1")
        self.__names_map[Counters.PSA_TX1_STEERING_0_TOTAL_CACHE_ACCESS] = \
            ("Number of TX PSA1 steering rules cache total access on pipe 0")
        self.__names_map[Counters.PSA_TX1_STEERING_1_TOTAL_CACHE_ACCESS] = \
            ("Number of TX PSA1 steering rules cache total access on pipe 1")
        self.__names_map[
            Counters.PSA_RX0_STEERING_PERF_COUNT_STEERING0_RSE_WORK_RATE] = \
            ("Number of RX PSA0 steering events on pipe 0")
        self.__names_map[
            Counters.PSA_RX0_STEERING_PERF_COUNT_STEERING1_RSE_WORK_RATE] = \
            ("Number of RX PSA0 steering events on pipe 1")
        self.__names_map[
            Counters.PSA_RX0_STEERING_PERF_COUNT_STEERING0_CACHE_HIT] = \
            ("Number of RX PSA0 steering rules cache hit on pipe 0")
        self.__names_map[
            Counters.PSA_RX0_STEERING_PERF_COUNT_STEERING0_CACHE_MISS] = \
            ("Number of RX PSA0 steering rules cache miss on pipe 0")
        self.__names_map[
            Counters.PSA_RX0_STEERING_PERF_COUNT_STEERING1_CACHE_HIT] = \
            ("Number of RX PSA0 steering rules cache hit on pipe 1")
        self.__names_map[
            Counters.PSA_RX0_STEERING_PERF_COUNT_STEERING1_CACHE_MISS] = \
            ("Number of RX PSA0 steering rules cache miss on pipe 1")
        self.__names_map[
            Counters.PSA_RX0_STEERING_PERF_COUNT_STEERING0_CACHE_TOT] = \
            ("Number of RX PSA0 steering rules cache total access on pipe 0")
        self.__names_map[
            Counters.PSA_RX0_STEERING_PERF_COUNT_STEERING1_CACHE_TOT] = \
            ("Number of RX PSA0 steering rules cache total access on pipe 1")
        self.__names_map[
            Counters.PSA_RX1_STEERING_PERF_COUNT_STEERING0_RSE_WORK_RATE] = \
            ("Number of RX PSA1 steering events on pipe 0")
        self.__names_map[
            Counters.PSA_RX1_STEERING_PERF_COUNT_STEERING1_RSE_WORK_RATE] = \
            ("Number of RX PSA1 steering events on pipe 1")
        self.__names_map[
            Counters.PSA_RX1_STEERING_PERF_COUNT_STEERING0_CACHE_HIT] = \
            ("Number of RX PSA1 steering rules cache hit on pipe 0")
        self.__names_map[
            Counters.PSA_RX1_STEERING_PERF_COUNT_STEERING0_CACHE_MISS] = \
            ("Number of RX PSA1 steering rules cache miss on pipe 0")
        self.__names_map[
            Counters.PSA_RX1_STEERING_PERF_COUNT_STEERING1_CACHE_HIT] = \
            ("Number of RX PSA1 steering rules cache hit on pipe 1")
        self.__names_map[
            Counters.PSA_RX1_STEERING_PERF_COUNT_STEERING1_CACHE_MISS] = \
            ("Number of RX PSA1 steering rules cache miss on pipe 1")
        self.__names_map[
            Counters.PSA_RX1_STEERING_PERF_COUNT_STEERING0_CACHE_TOT] = \
            ("Number of RX PSA1 steering rules cache total access on pipe 0")
        self.__names_map[
            Counters.PSA_RX1_STEERING_PERF_COUNT_STEERING1_CACHE_TOT] = \
            ("Number of RX PSA1 steering rules cache total access on pipe 1")

    def __initialize_map(self):
        """ Initialize of counters name map.
        """
        self.__initialize_tpt()
        self.__initialize_rxb()
        self.__initialize_ptb()
        self.__initialize_rxt()
        self.__initialize_rxw()
        self.__initialize_rxs()
        self.__initialize_rxc()
        self.__initialize_rxd()
        self.__initialize_sxw()
        self.__initialize_sxd()
        self.__initialize_rxps()
        self.__initialize_pxt()
        self.__initialize_sxp()
        self.__initialize_psa()

    def get_description(self, counter_name):
        """ Returns counter's description from its name.
        """
        return self.__names_map.get(counter_name, '')
