# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Simon Raviv
# @date: April 02, 2017

from mcra import MCRACounters
from performance.entities.counter import Counter
from performance.entities.unit import Unit
from performance.common.units import Units
from performance.common.units_measure import UnitsMeasure
from performance.common.device_ids import DeviceIDs
from performance.common.counters_names import CountersNames as Counters
from performance.common.utilization_reference import UtilizationReference \
    as Reference


class ConnectX4MCRACountres(MCRACounters):
    """ Represents ConnectX4 counters definitions.
    """

    def __init__(self):
        super(ConnectX4MCRACountres, self).__init__()
        self.units.append(Unit(Units.TPT, 0x0a6520))
        self.units.append(Unit(Units.PXDP, 0x1114a0))
        self.units.append(Unit(Units.PXT, 0x104b20))
        self.units.append(Unit(Units.RXB, 0x0dc420))
        self.units.append(Unit(Units.RXT, 0x0599a0))
        self.units.append(Unit(Units.RXW, 0x0650a0))
        self.units.append(Unit(Units.RXS, 0x0749a0))
        self.units.append(Unit(Units.RXC, 0x086920))
        self.units.append(Unit(Units.SXD, 0x016820))
        self.units.append(Unit(Units.SXP, 0x038aa0))
        self.units.append(Unit(Units.SXW, 0x02cd20))

        # Counters that are not defined yet, are commented out.
        # TPT #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=69,
                             name=Counters.TPT_L0_MTT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=68,
                             name=Counters.TPT_L0_MTT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=74,
                             name=Counters.TPT_L1_MTT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=73,
                             name=Counters.TPT_L1_MTT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=5,
                             utilization_bad=15))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=84,
                             name=Counters.TPT_L0_MPT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=83,
                             name=Counters.TPT_L0_MPT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=89,
                             name=Counters.TPT_L1_MPT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=88,
                             name=Counters.TPT_L1_MPT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=5,
                             utilization_bad=15))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=81,
                             name=Counters.TPT_INDIRECT_MEM_KEY,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=1,
                             utilization_bad=1))
        # PXT #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXT),
                             selector=134,
                             name=Counters.PXT_PXD_READY_BP,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXT),
                             selector=142,
                             name=Counters.PXT_PCI_WRITE_BP,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXT),
                             selector=146,
                             name=Counters.PXT_PCI_READ_BP,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXT),
                             selector=82,
                             name=Counters.
                             PXT_PCI_READ_STUCK_DUE_TO_NO_COMPLETION_BUFFER,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXT),
                             selector=78,
                             name=Counters.PXT_LACK_IN_CMP_ENG_BP,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXT),
                             selector=58,
                             name=Counters.PXT_PERF_RD_ICMC_PUSH_LINK0,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=1,
                             utilization_bad=5))
        # RXB #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=39,
                             name=Counters.RXB_RXS_NO_SLOW_PATH_CREDITS,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=1,
                             utilization_bad=5))
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.RXB),
        #    selector=40,
        #    name="RXB_RXS_NO_FAST_PATH_CREDITS",
        #    units=UnitsMeasure.CYCLES,
        #    utilization_reference=Reference.CYCLES,
        #    utilization_good=1,
        #    utilization_bad=5))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=35,
                             name=Counters.
                             RXB_RXT_NO_SLOW_PATH_CRED_PERF_COUNT,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=1,
                             utilization_bad=5))
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.RXB),
        #    selector=36,
        #    name="RXB_RXT_NO_FAST_PATH_CRED_PERF_COUNT",
        #    units=UnitsMeasure.CYCLES,
        #    utilization_reference=Reference.CYCLES,
        #    utilization_good=1,
        #    utilization_bad=5))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=46,
                             name=Counters.RXB_LRO_FIFO_PERF_COUNT2,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS,
                             utilization_good=1,
                             utilization_bad=5))
        # RXT #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXT),
                             selector=29,
                             name=Counters.RXT_CTRL_PERF_SLICE_LOAD_SLOW,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXT),
                             selector=28,
                             name=Counters.RXT_CTRL_PERF_SLICE_LOAD_FAST,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXT),
                             selector=49,
                             name=Counters.
                             RXT_STEERING_PERF_COUNT_STEERING0_RSE_WORK_RATE,
                             units=UnitsMeasure.EVENTS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXT),
                             selector=68,
                             name=Counters.
                             RXT_STEERING_PERF_COUNT_STEERING1_RSE_WORK_RATE,
                             units=UnitsMeasure.EVENTS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXT),
                             selector=52,
                             name=Counters.
                             RXT_STEERING_PERF_COUNT_STEERING0_CACHE_HIT,
                             units=UnitsMeasure.EVENTS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXT),
                             selector=51,
                             name=Counters.
                             RXT_STEERING_PERF_COUNT_STEERING0_CACHE_MISS,
                             units=UnitsMeasure.EVENTS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXT),
                             selector=71,
                             name=Counters.
                             RXT_STEERING_PERF_COUNT_STEERING1_CACHE_HIT,
                             units=UnitsMeasure.EVENTS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXT),
                             selector=70,
                             name=Counters.
                             RXT_STEERING_PERF_COUNT_STEERING1_CACHE_MISS,
                             units=UnitsMeasure.EVENTS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXT),
                             selector=50,
                             name=Counters.
                             RXT_STEERING_PERF_COUNT_STEERING0_CACHE_TOT,
                             units=UnitsMeasure.EVENTS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXT),
                             selector=69,
                             name=Counters.
                             RXT_STEERING_PERF_COUNT_STEERING1_CACHE_TOT,
                             units=UnitsMeasure.EVENTS))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXT),
        #                     selector=53,
        #                     name="RXT_STEERING_PERF_COUNT_PIPE0_FOLLOWERS"))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXT),
        #                     selector=72,
        #                     name="RXT_STEERING_PERF_COUNT_PIPE1_FOLLOWERS"))
        # RXW #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXW),
                             selector=167,
                             name=Counters.RXW_PERF_COUNT_TPT_CREDIT,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXW),
                             selector=51,
                             name=Counters.RXW_PERF_WB_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXW),
                             selector=52,
                             name=Counters.RXW_PERF_WB_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.RXW),
        #    selector=147,
        #    name="RXW_PERF_RX_L1_SLOW_HIT_LDB"))
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.RXW),
        #    selector=146,
        #    name="RXW_PERF_RX_L1_SLOW_MISS_LDB"))
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.RXW),
        #    selector=149,
        #    name="RXW_PERF_RX_L1_FAST_HIT_LDB"))
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.RXW),
        #    selector=148,
        #    name="RXW_PERF_RX_L1_FAST_MISS_LDB"))
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.RXW),
        #    selector=121,
        #    name="RXW_PERF_L2_CACHE_READ_HIT_LDB"))
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.RXW),
        #    selector=122,
        #    name="RXW_PERF_L2_CACHE_READ_MISS_LDB"))
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.RXW),
        #    selector=177,
        #    name="RXW_PERF_RX_L1_SLOW_HIT_REQSL"))
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.RXW),
        #    selector=176,
        #    name="RXW_PERF_RX_L1_SLOW_MISS_REQSL"))
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.RXW),
        #    selector=179,
        #    name="RXW_PERF_RX_L1_FAST_HIT_REQSL"))
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.RXW),
        #    selector=178,
        #    name="RXW_PERF_RX_L1_FAST_MISS_REQSL"))
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.RXW),
        #    selector=171,
        #    name="RXW_PERF_L2_CACHE_READ_HIT_REQSL"))
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.RXW),
        #    selector=172,
        #    name="RXW_PERF_L2_CACHE_READ_MISS_REQSL"))
        # RXS #
        # These two must be in this order.
        # When setting a selector to 0, even counters count wraparound
        # Odd counters count clocks.
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXS),
                             selector=0,
                             name=Counters.DEVICE_CLOCKS_WRAPAROUND))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXS),
                             selector=0,
                             name=Counters.DEVICE_CLOCKS,
                             units=UnitsMeasure.CYCLES))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXS),
                             selector=22,
                             name=Counters.RXS_NO_PXT_CREDITS,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        # RXC #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=104,
                             name=Counters.RXC_EQ_ALL_SLICES_BUSY,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=173,
                             name=Counters.RXC_CQ_ALL_SLICES_BUSY,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=43,
                             name=Counters.RXC_MSIX_ALL_SLICES_BUSY,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=219,
                             name=Counters.RXC_CQE_ZIP_OPEN_SESSION,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=220,
                             name=Counters.RXC_CQE_ZIP_MERGING_CQE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=221,
                             name=Counters.RXC_CQE_ZIP_WRITING_8_CQES,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=222,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_EQE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=223,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_TIMEOUT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=224,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_NOT_MATCH,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=225,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_PX_IDLE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=226,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_S2CQE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=227,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_LRO,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        # SXW #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=128,
                             name=Counters.SXW_QP_DONE_DUE_TO_VL_LIMITED,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))

        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=129,
                             name=Counters.SXW_QP_DONE_DUE_TO_DESCHED,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=130,
                             name=Counters.SXW_QP_DONE_DUE_TO_WORK_DONE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=10,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=131,
                             name=Counters.SXW_QP_DONE_DUE_TO_LIMITED,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=132,
                             name=Counters.SXW_QP_DONE_DUE_TO_E2E_CREDITS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=210,
                             name=Counters.SXW_PACKET_SEND_SXW2SXP_GO_VID,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(unit=self.get_unit(Units.SXW),
                             selector=243,
                             name=Counters.SXW_PERF_COUNT_STEERING_RSE0))
        self.counters.append(Counter(unit=self.get_unit(Units.SXW),
                             selector=236,
                             name=Counters.SXW_PERF_COUNT_STEERING_HIT,
                             units=UnitsMeasure.EVENTS))
        self.counters.append(Counter(unit=self.get_unit(Units.SXW),
                             selector=237,
                             name=Counters.SXW_PERF_COUNT_STEERING_MISS,
                             units=UnitsMeasure.EVENTS))
        self.counters.append(Counter(unit=self.get_unit(Units.SXW),
                             selector=238,
                             name=Counters.SXW_STEERING_PERF_COUNT_CACHE_TOT,
                             units=UnitsMeasure.EVENTS))
        self.counters.append(Counter(unit=self.get_unit(Units.SXW),
                             selector=239,
                             name=Counters.SXW_STEERING_PERF_COUNT_CACHE_MISS,
                             units=UnitsMeasure.EVENTS))
        self.counters.append(Counter(unit=self.get_unit(Units.SXW),
                             selector=240,
                             name=Counters.SXW_STEERING_PERF_COUNT_CACHE_HIT,
                             units=UnitsMeasure.EVENTS))
        # SXD #
        # self.counters.append(Counter(
        #    unit=self.get_unit(Units.SXD),
        #    selector=195,
        #    name="SXD_NO_PATH_CREDITS",
        #    units=UnitsMeasure.CYCLES,
        #    utilization_reference=Reference.CYCLES,
        #    utilization_good=10,
        #    utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXD),
                             selector=196,
                             name=Counters.SXD_NO_SLOW_PATH_CREDITS,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        # SXP #
        self.counters.append(Counter(unit=self.get_unit(Units.SXP),
                             selector=19,
                             name=Counters.
                             SXP_SXP_LINE_TRANSMITTED_PERF_COUNT0,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             TX_PACKETS,
                             utilization_good=100,
                             utilization_bad=100))
        # PXDP #
        self.counters.append(Counter(unit=self.get_unit(Units.PXDP),
                             selector=470,
                             name=Counters.PXDP_RX_128B_DATA))
        self.counters.append(Counter(unit=self.get_unit(Units.PXDP),
                             selector=469,
                             name=Counters.PXDP_RX_128B_TOTAL))

    @staticmethod
    def is_device_id_ok(id):
        if id in DeviceIDs.CONNECTX4:
            return True
        return False

    @staticmethod
    def set_latency_counters(mf):
        """ Set latency counters for sampling.
        """
        # Enable PCIe latency measurements:
        mf.write_field(val=1, addr=0x1025a8, startBit=0, size=1)

        # Set flavor to measuring PXT to response:
        mf.write_field(val=0, addr=0x10168c, startBit=31, size=1)

        # PXDP to response:
        # Reset the counters to NULL:
        mf.write_field(val=1, addr=0x104a80, startBit=0, size=1)
        mf.write_field(val=1, addr=0x104a80, startBit=1, size=1)
        mf.write_field(val=1, addr=0x104a80, startBit=2, size=1)

        # Release counter lock:
        mf.write_field(val=1, addr=0x104a84, startBit=0, size=2)

        # PXT to PXDP:
        # reset the counters to NULL
        mf.write_field(val=1, addr=0x104a00, startBit=0, size=1)
        mf.write_field(val=1, addr=0x104a00, startBit=1, size=1)
        mf.write_field(val=1, addr=0x104a00, startBit=2, size=1)

        # Enables the counter that will be used to measure latency:
        mf.write_field(val=1, addr=0x104a04, startBit=0, size=1)

        # Release counter lock"
        mf.write_field(val=1, addr=0x104a04, startBit=0, size=2)

    def get_latency_counters(self, mf, dev_clock):
        """ Returns latency counters.
        """
        # Reset the counters to NULL:
        mf.write_field(val=0, addr=0x104a80, startBit=0, size=1)
        mf.write_field(val=0, addr=0x104a80, startBit=1, size=1)
        mf.write_field(val=0, addr=0x104a80, startBit=2, size=1)
        mf.write_field(val=0, addr=0x104a00, startBit=0, size=1)
        mf.write_field(val=0, addr=0x104a00, startBit=1, size=1)
        mf.write_field(val=0, addr=0x104a00, startBit=2, size=1)

        # Lock counters for sum/cnt:
        mf.write_field(val=2, addr=0x104a84, startBit=0, size=2)
        mf.write_field(val=2, addr=0x104a04, startBit=0, size=2)

        # Read latency counters:
        link0_pxdp_to_pcie_response_latency_sum_31_0 = mf.read4(0x0104a9c)
        link0_pxdp_to_pcie_response_latency_sum_63_32 = mf.read4(0x0104a98)
        link0_pxdp_to_pcie_response_latency_cnt_31_0 = mf.read4(0x0104a94)
        link0_pxdp_to_pcie_response_min_latency = mf.read4(0x0104a88)
        link0_pxdp_to_pcie_response_max_latency = mf.read4(0x0104a8c)
        link0_pxt_to_pxdp_latency_sum_31_0 = mf.read4(0x0104a1c)
        link0_pxt_to_pxdp_latency_sum_63_32 = mf.read4(0x0104a18)
        link0_pxt_to_pxdp_latency_cnt_31_0 = mf.read4(0x0104a14)
        link0_pxt_to_pxdp_min_latency = mf.read4(0x0104a08)
        link0_pxt_to_pxdp_max_latency = mf.read4(0x0104a0c)

        pxdp_to_pcie_response_latency_link0 = 0
        if link0_pxdp_to_pcie_response_latency_cnt_31_0:
            pxdp_to_pcie_response_latency_cnt_total = \
                link0_pxdp_to_pcie_response_latency_sum_31_0 + \
                (link0_pxdp_to_pcie_response_latency_sum_63_32 << 32)
            pxdp_to_pcie_response_latency_link0 = \
                pxdp_to_pcie_response_latency_cnt_total * 1000 / \
                link0_pxdp_to_pcie_response_latency_cnt_31_0 / dev_clock

        pxdp_to_pcie_response_latency_link0_min = \
            link0_pxdp_to_pcie_response_min_latency * 1000 / dev_clock
        pxdp_to_pcie_response_latency_link0_max = \
            link0_pxdp_to_pcie_response_max_latency * 1000 / dev_clock

        pxt_to_pxdp_latency_link0 = 0
        if link0_pxt_to_pxdp_latency_cnt_31_0:
            pxt_to_pxdp_latency_sum_total = \
                link0_pxt_to_pxdp_latency_sum_31_0 + \
                (link0_pxt_to_pxdp_latency_sum_63_32 << 32)
            pxt_to_pxdp_latency_link0 = \
                pxt_to_pxdp_latency_sum_total * 1000 / \
                link0_pxt_to_pxdp_latency_cnt_31_0 / dev_clock

        pxt_to_pxdp_latency_link0_min = \
            link0_pxt_to_pxdp_min_latency * 1000 / dev_clock
        pxt_to_pxdp_latency_link0_max = \
            link0_pxt_to_pxdp_max_latency * 1000 / dev_clock

        # In case of counter overflow:
        if pxt_to_pxdp_latency_link0 == 0:
            pxt_to_pxdp_latency_link0_min = 0
            pxt_to_pxdp_latency_link0_max = 0

        if pxdp_to_pcie_response_latency_link0 == 0:
            pxdp_to_pcie_response_latency_link0_min = 0
            pxdp_to_pcie_response_latency_link0_max = 0

        # Initialize counters objects:
        self.regular_counters.append(
            Counter(name=Counters.PXT_TO_PXDP_LATENCY_LINK0_AVG,
                    value=pxt_to_pxdp_latency_link0))
        self.regular_counters.append(
            Counter(name=Counters.PXT_TO_PXDP_LATENCY_LINK0_MIN,
                    value=pxt_to_pxdp_latency_link0_min))
        self.regular_counters.append(
            Counter(name=Counters.PXT_TO_PXDP_LATENCY_LINK0_MAX,
                    value=pxt_to_pxdp_latency_link0_max))
        self.regular_counters.append(
            Counter(name=Counters.PCI_LATENCY_LINK0_AVG,
                    value=pxdp_to_pcie_response_latency_link0))
        self.regular_counters.append(
            Counter(name=Counters.PCI_LATENCY_LINK0_MIN,
                    value=pxdp_to_pcie_response_latency_link0_min))
        self.regular_counters.append(
            Counter(name=Counters.PCI_LATENCY_LINK0_MAX,
                    value=pxdp_to_pcie_response_latency_link0_max))
