# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# @author: Simon Raviv
# @date: May 25, 2017

from mcra import MCRACounters
from performance.entities.unit import Unit
from performance.entities.counter import Counter
from performance.common.units import Units
from performance.common.units_measure import UnitsMeasure
from performance.common.device_ids import DeviceIDs
from performance.common.counters_names import CountersNames as Counters
from performance.common.utilization_reference import UtilizationReference \
        as Reference


class ConnectX5MCRACountres(MCRACounters):
    """ Represents ConnectX4LX counters definitions.
    """

    def __init__(self, device):
        super(ConnectX5MCRACountres, self).__init__(device=device)

        self._pci_link_gen_address = 0x137064
        self.__get_linkm_enable_bit_address = 0xf8cc

        self.units.append(Unit(Units.TPT, 0x0a6620))
        if self._check_pci_integrity_issue() is False:
            self.units.append(Unit(Units.PXDP, 0x126420))
        self.units.append(Unit(Units.PXT, 0x10faa0))
        self.units.append(Unit(Units.RXB, 0x0dc420))
        self.units.append(Unit(Units.RXT, 0x05a220))
        self.units.append(Unit(Units.RXW, 0x0650a0))
        self.units.append(Unit(Units.RXS, 0x0749a0))
        self.units.append(Unit(Units.RXC, 0x086a20))
        self.units.append(Unit(Units.SXP, 0x0389a0))
        self.units.append(Unit(Units.SXP_E2E_CACHE,
                               0x039650, 8, 0x039654, 0x039640, 8))
        self.units.append(Unit(Units.SXW, 0x02cd20))
        self.units.append(Unit(Units.SXD, 0x016820))
        self.units.append(Unit(Units.RXPS, 0x043020))
        self.units.append(Unit(Units.RXPS_E2E_CACHE,
                               0x040050, 8, 0x040054, 0x040040, 8))

        # Counters that are not defined yet, are commented out.
        # TPT #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=258,
                             name=Counters.TPT_L0_MTT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=257,
                             name=Counters.TPT_L0_MTT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=263,
                             name=Counters.TPT_L1_MTT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=262,
                             name=Counters.TPT_L1_MTT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=5,
                             utilization_bad=15))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=273,
                             name=Counters.TPT_L0_MPT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=272,
                             name=Counters.TPT_L0_MPT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=278,
                             name=Counters.TPT_L1_MPT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=277,
                             name=Counters.TPT_L1_MPT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=5,
                             utilization_bad=15))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=270,
                             name=Counters.TPT_INDIRECT_MEM_KEY,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=1,
                             utilization_bad=1))

        # PXT #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXT),
                             selector=399,
                             name=Counters.PXT_PXD_READY_BP,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=1,
                             utilization_bad=5))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXT),
                             selector=423,
                             name=Counters.PXT_PCI_READ_BP,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXT),
                             selector=415,
                             name=Counters.PXT_PCI_WRITE_BP,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXT),
                             selector=255,
                             name=Counters.PXT_PERF_RD_ICMC_PUSH_LINK0,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=1,
                             utilization_bad=5))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXT),
                             selector=263,
                             name=Counters.
                             PXT_PCI_READ_STUCK_DUE_TO_NO_READ_ENGINES,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXT),
                             selector=271,
                             name=Counters.
                             PXT_PCI_READ_STUCK_DUE_TO_BYTE_LIMIT,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXT),
                             selector=287,
                             name=Counters.PXT_PCI_READ_STUCK_DUE_TO_ORDERING,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=9,
                             name=Counters.RXB_BW_COUNT_PERF_COUNT_1_1,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=10,
                             name=Counters.RXB_BW_COUNT_PERF_COUNT_1_2,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=12,
                             name=Counters.RXB_BW_COUNT_PERF_COUNT_0_1,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=13,
                             name=Counters.RXB_BW_COUNT_PERF_COUNT_0_2,
                             units=UnitsMeasure.PACKETS))
        # RXB #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=52,
                             name=Counters.RXB_RXS_NO_SLOW_PATH_CREDITS,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))

        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXB),
        #                     selector=53,
        #                     name="RXB_RXS_NO_FAST_PATH_CREDITS",
        #                     units=UnitsMeasure.CYCLES,
        #                     utilization_reference=Reference.CYCLES,
        #                     utilization_good=10,
        #                     utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=48,
                             name=Counters.
                             RXB_RXT_NO_SLOW_PATH_CRED_PERF_COUNT,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXB),
        #                     selector=49,
        #                     name="RXB_RXT_NO_FAST_PATH_CRED_PERF_COUNT",
        #                     units=UnitsMeasure.CYCLES,
        #                     utilization_reference=Reference.CYCLES,
        #                     utilization_good=10,
        #                     utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=17,
                             name=Counters.
                             RXB_PORT0_BUFFER_FULL_PERF_COUNT_PORT0,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=15,
                             name=Counters.
                             RXB_PORT1_BUFFER_FULL_PERF_COUNT_PORT1,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=1,
                             utilization_bad=10))

        # RXPS #
        # These two must be in this order. When setting a to 0,
        # even counters count wraparound Odd counters count clocks.
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=0,
                             name=Counters.DEVICE_CLOCKS_WRAPAROUND,
                             units=UnitsMeasure.CYCLES))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=0,
                             name=Counters.DEVICE_CLOCKS,
                             units=UnitsMeasure.CYCLES))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=26,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING0_RSE_WORK_RATE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=45,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING1_RSE_WORK_RATE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=30,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING0_CACHE_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=29,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING0_CACHE_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=49,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING1_CACHE_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=48,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING1_CACHE_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=28,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING0_CACHE_TOT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=47,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING1_CACHE_TOT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS_E2E_CACHE),
                             selector=15,
                             name=Counters.RXPS_E2E_CACHE_TOTAL_LOOKUPS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS_E2E_CACHE),
                             selector=24,
                             name=Counters.RXPS_E2E_CACHE_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS_E2E_CACHE),
                             selector=33,
                             name=Counters.RXPS_E2E_CACHE_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS,
                             utilization_good=5,
                             utilization_bad=15))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS_E2E_CACHE),
                             selector=42,
                             name=Counters.RXPS_E2E_CACHE_LEARN,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS,
                             utilization_good=1,
                             utilization_bad=1))

        # RXW #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXW),
                             selector=70,
                             name=Counters.RXW_PERF_COUNT_TPT_CREDIT,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXW),
                             selector=66,
                             name=Counters.RXW_PERF_WB_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXW),
                             selector=65,
                             name=Counters.RXW_PERF_WB_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXW),
        #                     selector=180,
        #                     name="RXW_PERF_L2_CACHE_READ_HIT_REQSL"))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXW),
        #                     selector=181,
        #                     name="RXW_PERF_L2_CACHE_READ_MISS_REQSL"))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXW),
        #                     selector=186,
        #                     name="RXW_PERF_RX_L1_SLOW_HIT_REQSL"))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXW),
        #                     selector=185,
        #                     name="RXW_PERF_RX_L1_SLOW_MISS_REQSL"))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXW),
        #                     selector=188,
        #                     name="RXW_PERF_RX_L1_FAST_HIT_REQSL"))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXW),
        #                     selector=187,
        #                     name="RXW_PERF_RX_L1_FAST_MISS_REQSL"))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXW),
        #                     selector=11,
        #                     name="RXW_PERF_RXS_ATOMIC_FIFO_0"))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXW),
        #                     selector=12,
        #                     name="RXW_PERF_RXS_ATOMIC_FIFO_1"))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXW),
        #                     selector=14,
        #                     name="RXW_PERF_RXS_ATOMIC_CRED"))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXW),
        #                     selector=121,
        #                     name="RXW_PERF_RX_L1_HIT_LDB"))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXW),
        #                     selector=119,
        #                     name="RXW_PERF_RX_L1_MISS_LDB"))

        # RXS #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXS),
                             selector=23,
                             name=Counters.RXS_NO_PXT_CREDITS,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))

        # RXT #
        self.counters.append(Counter(unit=self.get_unit(Units.RXT),
                             selector=21,
                             name=Counters.RXT_CTRL_PERF_SLICE_LOAD_SLOW,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(unit=self.get_unit(Units.RXT),
                             selector=20,
                             name=Counters.RXT_CTRL_PERF_SLICE_LOAD_FAST,
                             units=UnitsMeasure.PACKETS))

        # PXDP #
        if self._check_pci_integrity_issue() is False:
            self.counters.append(Counter(
                                 unit=self.get_unit(Units.PXDP),
                                 selector=723,
                                 name=Counters.PXDP_RX_128B_TOTAL))
            self.counters.append(Counter(
                                 unit=self.get_unit(Units.PXDP),
                                 selector=724,
                                 name=Counters.PXDP_RX_128B_DATA))
            self.counters.append(Counter(
                                 unit=self.get_unit(Units.PXDP),
                                 selector=650,
                                 name=Counters.PXDP_TX_128B_DATA))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.PXDP),
        #                     selector=684,
        #                     name="PXDP_RX_128B_TOTAL_LINK4",
        #                     units=UnitsMeasure.BYTES))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.PXDP),
        #                     selector=685,
        #                     name="PXDP_RX_128B_DATA_LINK4",
        #                     units=UnitsMeasure.BYTES,
        #                     utilization_reference=Reference.
        #                     AVAILABLE_RX_LINK_4_PCIE_BANDWIDTH,
        #                     utilization_good=95,
        #                     utilization_bad=99))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.PXDP),
        #                     selector=684,
        #                     name="PXDP_RX_128B_TOTAL_LINK8",
        #                     units=UnitsMeasure.BYTES))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.PXDP),
        #                     selector=685,
        #                     name="PXDP_RX_128B_DATA_LINK8",
        #                     units=UnitsMeasure.BYTES,
        #                     utilization_reference=Reference.
        #                     AVAILABLE_RX_LINK_8_PCIE_BANDWIDTH,
        #                     utilization_good=95,
        #                     utilization_bad=99))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.PXDP),
        #                     selector=684,
        #                     name="PXDP_RX_128B_TOTAL_LINK12",
        #                     units=UnitsMeasure.BYTES))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.PXDP),
        #                     selector=685,
        #                     name="PXDP_RX_128B_DATA_LINK12",
        #                     units=UnitsMeasure.BYTES,
        #                     utilization_reference=Reference.
        #                     AVAILABLE_RX_LINK_12_PCIE_BANDWIDTH,
        #                     utilization_good=95,
        #                     utilization_bad=99))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.PXDP),
        #                     selector=650,
        #                     name="PXDP_TX_128B_DATA",
        #                     units=UnitsMeasure.BYTES,
        #                     utilization_reference=Reference.
        #                     AVAILABLE_PCIE_BANDWIDTH,
        #                     utilization_good=95,
        #                     utilization_bad=99))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.PXDP),
        #                     selector=612,
        #                     name="PXDP_TX_128B_TOTAL_LINK4",
        #                     units=UnitsMeasure.BYTES))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.PXDP),
        #                     selector=611,
        #                     name="PXDP_TX_128B_DATA_LINK4",
        #                     units=UnitsMeasure.BYTES,
        #                     utilization_reference=Reference.
        #                     AVAILABLE_TX_LINK_4_PCIE_BANDWIDTH,
        #                     utilization_good=95,
        #                     utilization_bad=99))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.PXDP),
        #                     selector=612,
        #                     name="PXDP_TX_128B_TOTAL_LINK8",
        #                     units=UnitsMeasure.BYTES))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.PXDP),
        #                     selector=611,
        #                     name="PXDP_TX_128B_DATA_LINK8",
        #                     units=UnitsMeasure.BYTES,
        #                     utilization_reference=Reference.
        #                     AVAILABLE_TX_LINK_8_PCIE_BANDWIDTH,
        #                     utilization_good=95,
        #                     utilization_bad=99))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.PXDP),
        #                     selector=612,
        #                     name="PXDP_TX_128B_TOTAL_LINK12",
        #                     units=UnitsMeasure.BYTES))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.PXDP),
        #                     selector=611,
        #                     name="PXDP_TX_128B_DATA_LINK12",
        #                     units=UnitsMeasure.BYTES,
        #                     utilization_reference=Reference.
        #                     AVAILABLE_TX_LINK_12_PCIE_BANDWIDTH,
        #                     utilization_good=95,
        #                     utilization_bad=99))

        # RXC #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=104,
                             name=Counters.RXC_EQ_ALL_SLICES_BUSY,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=180,
                             name=Counters.RXC_CQ_ALL_SLICES_BUSY,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=43,
                             name=Counters.RXC_MSIX_ALL_SLICES_BUSY,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=227,
                             name=Counters.RXC_CQE_ZIP_OPEN_SESSION,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS,
                             utilization_good=15,
                             utilization_bad=50))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXC),
        #                     selector=260,
        #                     name="RXC_CQE_ZIP_REPLACE_SESSION"))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=228,
                             name=Counters.RXC_CQE_ZIP_MERGING_CQE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=230,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_EQE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=231,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_TIMEOUT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=232,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_NOT_MATCH,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=233,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_PX_IDLE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=234,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_S2CQE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=229,
                             name=Counters.RXC_CQE_ZIP_WRITING_8_CQES,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=235,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_LRO,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))

        # SXW #
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.SXW),
        #                     selector=80,
        #                     name="SXW_PERF_CNT_QUERY_WITH_HIT"))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.SXW),
        #                     selector=81,
        #                     name="SXW_PERF_CNT_QUERY_WITH_MISS"))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=161,
                             name=Counters.SXW_QP_DONE_DUE_TO_LIMITED,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=158,
                             name=Counters.SXW_QP_DONE_DUE_TO_VL_LIMITED,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=159,
                             name=Counters.SXW_QP_DONE_DUE_TO_DESCHED,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=160,
                             name=Counters.SXW_QP_DONE_DUE_TO_WORK_DONE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=10,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=162,
                             name=Counters.SXW_QP_DONE_DUE_TO_E2E_CREDITS,
                             units=UnitsMeasure.EVENTS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=242,
                             name=Counters.SXW_PACKET_SEND_SXW2SXP_GO_VID,
                             units=UnitsMeasure.PACKETS))
        # SXD #
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.SXD),
        #                     selector=439,
        #                     name="SXD_SXW_RETURN_QP_AS_WAREHOUSE",
        #                     units=UnitsMeasure.EVENTS,
        #                     utilization_reference=Reference.TX_PACKETS))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.SXD),
        #                     selector=438,
        #                     name="SXD_SXW_RETURN_QP_AS_LIMITED",
        #                     units=UnitsMeasure.EVENTS,
        #                     utilization_reference=Reference.TX_PACKETS,
        #                     utilization_good=1,
        #                     utilization_bad=10))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.SXD),
        #                     selector=436,
        #                     name="SXD_SXW_RETURN_QP_AS_PI_EQUAL_CI",
        #                     units=UnitsMeasure.EVENTS,
        #                     utilization_reference=Reference.TX_PACKETS,
        #                     utilization_good=1,
        #                     utilization_bad=10))
        # self.counters.append(Counter(
        #                     unit=self.get_unit(Units.SXD),
        #                     selector=437,
        #                     name="SXD_SXW_RETURN_QP_AS_OUT_OF_TOKENS",
        #                     units=UnitsMeasure.EVENTS,
        #                     utilization_reference=Reference.TX_PACKETS,
        #                     utilization_good=1,
        #                     utilization_bad=10))
        # SXP #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=139,
                             name=Counters.SXP_LINE_TRANSMITTED_PORT0,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=140,
                             name=Counters.SXP_LINE_TRANSMITTED_PORT1,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=141,
                             name=Counters.SXP_LINE_TRANSMITTED_LOOPBACK,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=29,
                             name=Counters.
                             SXP_STEERING_PERF_COUNT_STEERING0_RSE_WORK_RATE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=27,
                             name=Counters.
                             SXP_STEERING_PERF_COUNT_STEERING1_RSE_WORK_RATE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=14,
                             name=Counters.SXP_PERF_COUNT_STEERING0_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=16,
                             name=Counters.SXP_PERF_COUNT_STEERING0_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=13,
                             name=Counters.SXP_PERF_COUNT_STEERING1_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=15,
                             name=Counters.SXP_PERF_COUNT_STEERING1_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=23,
                             name=Counters.
                             SXP_PERF_COUNT_STEERING0_CACHE_1_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=18,
                             name=Counters.
                             SXP_PERF_COUNT_STEERING1_CACHE_1_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=22,
                             name=Counters.SXP_STEERING_0_TOTAL_CACHE_ACCESS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=17,
                             name=Counters.SXP_STEERING_1_TOTAL_CACHE_ACCESS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP_E2E_CACHE),
                             selector=15,
                             name=Counters.SXP_E2E_CACHE_TOTAL_LOOKUPS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP_E2E_CACHE),
                             selector=24,
                             name=Counters.SXP_E2E_CACHE_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP_E2E_CACHE),
                             selector=33,
                             name=Counters.SXP_E2E_CACHE_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=5,
                             utilization_bad=15))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP_E2E_CACHE),
                             selector=42,
                             name=Counters.SXP_E2E_CACHE_LEARN,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=1))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=182,
                             name=Counters.SXP_BW_COUNT_PERF_COUNT_0_1))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=183,
                             name=Counters.SXP_BW_COUNT_PERF_COUNT_0_2))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=179,
                             name=Counters.SXP_BW_COUNT_PERF_COUNT_1_1))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=180,
                             name=Counters.SXP_BW_COUNT_PERF_COUNT_1_2))

    @staticmethod
    def is_device_id_ok(id):
        if id in (DeviceIDs.CONNECTX5 + DeviceIDs.CONNECTX5EX):
            return True
        return False

    @staticmethod
    def set_latency_counters(mf):
        """ Set latency counters for sampling.
        """
        # Enable PCIe latency measurements:
        mf.write_field(val=1, addr=0x10d6fc, startBit=0, size=1)

        # Set flavor to measuring PXT to response:
        mf.write_field(val=1, addr=0x103440, startBit=0, size=1)

        # PXT to response:
        # Reset the counters to NULL:
        mf.write_field(val=1, addr=0x10d900, startBit=0, size=1)
        mf.write_field(val=1, addr=0x10d900, startBit=1, size=1)
        mf.write_field(val=1, addr=0x10d900, startBit=2, size=1)

        # Enables the counter that will be used to measure latency:
        mf.write_field(val=1, addr=0x10d904, startBit=0, size=1)

        # Release counter lock:
        mf.write_field(val=0, addr=0x10d904, startBit=1, size=1)

        # PXT to PXDP:
        # Reset the counters to NULL:
        mf.write_field(val=1, addr=0x10d700, startBit=0, size=1)
        mf.write_field(val=1, addr=0x10d700, startBit=1, size=1)
        mf.write_field(val=1, addr=0x10d700, startBit=2, size=1)

        # Enables the counter that will be used to measure latency:
        mf.write_field(val=1, addr=0x10d704, startBit=0, size=1)

        # Release counter lock:
        mf.write_field(val=0, addr=0x10d704, startBit=1, size=1)

    def get_latency_counters(self, mf, dev_clock):
        """ Returns latency counters.
        """
        # Reset the counters to NULL:
        mf.write_field(val=0, addr=0x10d900, startBit=0, size=1)
        mf.write_field(val=0, addr=0x10d900, startBit=1, size=1)
        mf.write_field(val=0, addr=0x10d900, startBit=2, size=1)
        mf.write_field(val=0, addr=0x10d700, startBit=0, size=1)
        mf.write_field(val=0, addr=0x10d700, startBit=1, size=1)
        mf.write_field(val=0, addr=0x10d700, startBit=2, size=1)

        # Lock counters for sum/cnt:
        mf.write_field(val=0, addr=0x10d904, startBit=0, size=1)
        mf.write_field(val=1, addr=0x10d904, startBit=1, size=1)
        mf.write_field(val=0, addr=0x10d704, startBit=0, size=1)
        mf.write_field(val=1, addr=0x10d704, startBit=1, size=1)

        # Read latency counters:
        link0_pxt_to_pcie_response_latency_sum_31_0 = mf.read4(0x010d91c)
        link0_pxt_to_pcie_response_latency_sum_63_32 = mf.read4(0x010d918)
        link0_pxt_to_pcie_response_latency_cnt_31_0 = mf.read4(0x010d914)
        link0_pxt_to_pcie_response_min_latency = mf.read4(0x010d908)
        link0_pxt_to_pcie_response_max_latency = mf.read4(0x010d90c)
        link0_pxt_to_pxdp_latency_sum_31_0 = mf.read4(0x010d71c)
        link0_pxt_to_pxdp_latency_sum_63_32 = mf.read4(0x010d718)
        link0_pxt_to_pxdp_latency_cnt_31_0 = mf.read4(0x010d714)
        link0_pxt_to_pxdp_min_latency = mf.read4(0x010d708)
        link0_pxt_to_pxdp_max_latency = mf.read4(0x010d70c)

        pxt_to_pcie_response_latency_link0 = 0
        if link0_pxt_to_pcie_response_latency_cnt_31_0:
            pxt_to_pcie_response_latency_sum_total = \
                link0_pxt_to_pcie_response_latency_sum_31_0 + \
                (link0_pxt_to_pcie_response_latency_sum_63_32 << 32)
            pxt_to_pcie_response_latency_link0 = \
                pxt_to_pcie_response_latency_sum_total * 1000 / \
                link0_pxt_to_pcie_response_latency_cnt_31_0 / dev_clock
        pxt_to_pcie_response_latency_link0_min = \
            link0_pxt_to_pcie_response_min_latency * 1000 / dev_clock
        pxt_to_pcie_response_latency_link0_max = \
            link0_pxt_to_pcie_response_max_latency * 1000 / dev_clock

        pxt_to_pxdp_latency_link0 = 0
        if link0_pxt_to_pxdp_latency_cnt_31_0:
            pxt_to_pxdp_latency_sum_total = \
                link0_pxt_to_pxdp_latency_sum_31_0 + \
                (link0_pxt_to_pxdp_latency_sum_63_32 << 32)
            pxt_to_pxdp_latency_link0 = \
                pxt_to_pxdp_latency_sum_total * 1000 / \
                link0_pxt_to_pxdp_latency_cnt_31_0 / dev_clock
        pxt_to_pxdp_latency_link0_min = \
            link0_pxt_to_pxdp_min_latency * 1000 / dev_clock
        pxt_to_pxdp_latency_link0_max = \
            link0_pxt_to_pxdp_max_latency * 1000 / dev_clock

        # In case of counter overflow:
        if pxt_to_pxdp_latency_link0 == 0:
            pxt_to_pxdp_latency_link0_min = 0
            pxt_to_pxdp_latency_link0_max = 0

        if pxt_to_pcie_response_latency_link0 == 0:
            pxt_to_pcie_response_latency_link0_min = 0
            pxt_to_pcie_response_latency_link0_max = 0

        # Initialize counters objects:
        self.regular_counters.append(
            Counter(name=Counters.PXT_TO_PXDP_LATENCY_LINK0_AVG,
                    value=pxt_to_pxdp_latency_link0))
        self.regular_counters.append(
            Counter(name=Counters.PXT_TO_PXDP_LATENCY_LINK0_MIN,
                    value=pxt_to_pxdp_latency_link0_min))
        self.regular_counters.append(
            Counter(name=Counters.PXT_TO_PXDP_LATENCY_LINK0_MAX,
                    value=pxt_to_pxdp_latency_link0_max))
        self.regular_counters.append(
            Counter(name=Counters.PCI_LATENCY_LINK0_AVG,
                    value=(pxt_to_pcie_response_latency_link0 -
                           pxt_to_pxdp_latency_link0)))
        self.regular_counters.append(
            Counter(name=Counters.PCI_LATENCY_LINK0_MIN,
                    value=(pxt_to_pcie_response_latency_link0_min -
                           pxt_to_pxdp_latency_link0_min)))
        self.regular_counters.append(
            Counter(name=Counters.PCI_LATENCY_LINK0_MAX,
                    value=(pxt_to_pcie_response_latency_link0_max -
                           pxt_to_pxdp_latency_link0_max)))

    def __get_linkm_enable_bit(self):
        """ Return PCI linkm bit status.
        """
        # This bit can be set to 0, in order to disable the
        # FW to rely on PCI units performance counters.
        # For some flows, this can cause PCI integrity issues
        # in some scenarios.
        linkm_bit = self.device.read_field(
            addr=self.__get_linkm_enable_bit_address,
            startBit=16,
            size=1)
        return linkm_bit

    def _check_pci_integrity_issue(self):
        """ Check FW PCI integrity issues.
            This method intendeds to guard from FW PCI
            and the performance plugin to use the same counters
            in PCI units, in which the FW rely on in some scenarios.
            The issue will occur based on the condition below.
            In order to override the issue, user can set to 0 the linkm bit.
        """
        if (self._get_pci_link_gen() == 4
                and self.__get_linkm_enable_bit() == 1):
            return True
        else:
            return False
