# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --

# @author: Simon Raviv
# @date: September 04, 2018

from mcra import MCRACounters
from performance.entities.unit import Unit
from performance.entities.counter import Counter
from performance.common.units import Units
from performance.common.units_measure import UnitsMeasure
from performance.common.device_ids import DeviceIDs
from performance.common.counters_names import CountersNames as Counters
from performance.common.utilization_reference import UtilizationReference \
        as Reference


class ConnectX6MCRACountres(MCRACounters):
    """ Represents ConnectX-6 counters definitions.
    """

    def __init__(self, device):
        super(ConnectX6MCRACountres, self).__init__(device=device)

        self.units.append(Unit(name=Units.TPT,
                               en_addr=0xa6120,
                               en_start_bit=30,
                               start_addr=0xa6124,
                               selector_addr=0xa6100,
                               size=16))
        self.units.append(Unit(name=Units.PXDP,
                               en_addr=0x144020,
                               en_start_bit=30,
                               start_addr=0x144024,
                               selector_addr=0x144000,
                               size=16))
        self.units.append(Unit(name=Units.PXTL,
                               en_addr=0x10e420,
                               en_start_bit=30,
                               start_addr=0x10e424,
                               selector_addr=0x10e400,
                               size=16))
        self.units.append(Unit(name=Units.PXTH,
                               en_addr=0x20f020,
                               en_start_bit=30,
                               start_addr=0x20f024,
                               selector_addr=0x20f000,
                               size=16))
        self.units.append(Unit(name=Units.RXB,
                               en_addr=0x268720,
                               en_start_bit=30,
                               start_addr=0x268724,
                               selector_addr=0x268700,
                               size=16))
        self.units.append(Unit(name=Units.RXT,
                               en_addr=0x5a220,
                               en_start_bit=30,
                               start_addr=0x5a224,
                               selector_addr=0x5a200,
                               size=16))
        self.units.append(Unit(name=Units.RXW,
                               en_addr=0x65420,
                               en_start_bit=30,
                               start_addr=0x65424,
                               selector_addr=0x65400,
                               size=16))
        self.units.append(Unit(name=Units.RXS,
                               en_addr=0x74920,
                               en_start_bit=30,
                               start_addr=0x74924,
                               selector_addr=0x74900,
                               size=16))
        self.units.append(Unit(name=Units.RXC,
                               en_addr=0x86e20,
                               en_start_bit=30,
                               start_addr=0x86e24,
                               selector_addr=0x86e00,
                               size=16))
        self.units.append(Unit(name=Units.SXP,
                               en_addr=0x38920,
                               en_start_bit=30,
                               start_addr=0x38924,
                               selector_addr=0x38900,
                               size=16))
        self.units.append(Unit(name=Units.SXS_E2E_CACHE,
                               en_addr=0x44720,
                               en_start_bit=30,
                               start_addr=0x44724,
                               selector_addr=0x44700,
                               size=16))
        self.units.append(Unit(name=Units.SXW,
                               en_addr=0x2cd20,
                               en_start_bit=30,
                               start_addr=0x2cd24,
                               selector_addr=0x2cd00,
                               size=16))
        self.units.append(Unit(name=Units.SXD,
                               en_addr=0x1d620,
                               en_start_bit=30,
                               start_addr=0x1d624,
                               selector_addr=0x1d600,
                               size=16))
        self.units.append(Unit(name=Units.SXS,
                               en_addr=0x46420,
                               en_start_bit=30,
                               start_addr=0x46424,
                               selector_addr=0x46400,
                               size=16))
        self.units.append(Unit(name=Units.RXPS,
                               en_addr=0x43220,
                               en_start_bit=30,
                               start_addr=0x43224,
                               selector_addr=0x43200,
                               size=16))
        self.units.append(Unit(name=Units.RXPS_E2E_CACHE,
                               en_addr=0x40120,
                               en_start_bit=30,
                               start_addr=0x40124,
                               selector_addr=0x40100,
                               size=16))

        # TPT #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=77,
                             name=Counters.TPT_L0_MTT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=76,
                             name=Counters.TPT_L0_MTT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=82,
                             name=Counters.TPT_L1_MTT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=81,
                             name=Counters.TPT_L1_MTT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=5,
                             utilization_bad=15))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=55,
                             name=Counters.TPT_L0_MPT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=54,
                             name=Counters.TPT_L0_MPT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=60,
                             name=Counters.TPT_L1_MPT_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=59,
                             name=Counters.TPT_L1_MPT_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=5,
                             utilization_bad=15))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.TPT),
                             selector=71,
                             name=Counters.TPT_INDIRECT_MEM_KEY,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=1,
                             utilization_bad=1))
        # PXTH #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXTH),
                             selector=0,
                             name=Counters.PXT_PERF_RD_ICMC_PUSH_LINK0,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TOTAL_PACKETS,
                             utilization_good=1,
                             utilization_bad=5))
        # PXTL #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXTL),
                             selector=270,
                             name=Counters.PXT_PXD_READY_BP,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=1,
                             utilization_bad=5))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXTL),
                             selector=246,
                             name=Counters.PXT_PCI_READ_BP,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXTL),
                             selector=254,
                             name=Counters.PXT_PCI_WRITE_BP,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXTL),
                             selector=446,
                             name=Counters.
                             PXT_PCI_READ_STUCK_DUE_TO_NO_READ_ENGINES,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXTL),
                             selector=438,
                             name=Counters.
                             PXT_PCI_READ_STUCK_DUE_TO_BYTE_LIMIT,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXTL),
                             selector=406,
                             name=Counters.PXT_PCI_READ_STUCK_DUE_TO_ORDERING,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        # RXB #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=57,
                             name=Counters.RXB_BW_COUNT_PERF_COUNT_1_1,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=56,
                             name=Counters.RXB_BW_COUNT_PERF_COUNT_1_2,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=55,
                             name=Counters.RXB_BW_COUNT_PERF_COUNT_0_1,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=13,
                             name=Counters.RXB_BW_COUNT_PERF_COUNT_0_2,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=15,
                             name=Counters.RXB_RXS_NO_SLOW_PATH_CREDITS,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=19,
                             name=Counters.
                             RXB_RXT_NO_SLOW_PATH_CRED_PERF_COUNT,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=50,
                             name=Counters.
                             RXB_PORT0_BUFFER_FULL_PERF_COUNT_PORT0,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXB),
                             selector=52,
                             name=Counters.
                             RXB_PORT1_BUFFER_FULL_PERF_COUNT_PORT1,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=1,
                             utilization_bad=10))
        # RXPS #
        # These two must be in this order. When setting a to 0,
        # even counters count wraparound Odd counters count clocks.
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=0,
                             name=Counters.DEVICE_CLOCKS_WRAPAROUND,
                             units=UnitsMeasure.CYCLES))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=0,
                             name=Counters.DEVICE_CLOCKS,
                             units=UnitsMeasure.CYCLES))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=72,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING0_RSE_WORK_RATE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=53,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING1_RSE_WORK_RATE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=69,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING0_CACHE_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=68,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING0_CACHE_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=50,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING1_CACHE_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=49,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING1_CACHE_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=67,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING0_CACHE_TOT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS),
                             selector=48,
                             name=Counters.
                             RXPS_STEERING_PERF_COUNT_STEERING1_CACHE_TOT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS_E2E_CACHE),
                             selector=15,
                             name=Counters.RXPS_E2E_CACHE_TOTAL_LOOKUPS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS_E2E_CACHE),
                             selector=24,
                             name=Counters.RXPS_E2E_CACHE_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS_E2E_CACHE),
                             selector=33,
                             name=Counters.RXPS_E2E_CACHE_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS,
                             utilization_good=5,
                             utilization_bad=15))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXPS_E2E_CACHE),
                             selector=42,
                             name=Counters.RXPS_E2E_CACHE_LEARN,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS,
                             utilization_good=1,
                             utilization_bad=1))
        # RXW #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXW),
                             selector=98,
                             name=Counters.RXW_PERF_COUNT_TPT_CREDIT,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXW),
                             selector=102,
                             name=Counters.RXW_PERF_WB_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXW),
                             selector=103,
                             name=Counters.RXW_PERF_WB_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        # RXS #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXS),
                             selector=37,
                             name=Counters.RXS_NO_PXT_CREDITS,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        # RXT #
        self.counters.append(Counter(unit=self.get_unit(Units.RXT),
                             selector=21,
                             name=Counters.RXT_CTRL_PERF_SLICE_LOAD_SLOW,
                             units=UnitsMeasure.PACKETS))
        self.counters.append(Counter(unit=self.get_unit(Units.RXT),
                             selector=20,
                             name=Counters.RXT_CTRL_PERF_SLICE_LOAD_FAST,
                             units=UnitsMeasure.PACKETS))
        # PXDP #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXDP),
                             selector=50,
                             name=Counters.PXDP_RX_128B_TOTAL))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXDP),
                             selector=49,
                             name=Counters.PXDP_RX_128B_DATA))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.PXDP),
                             selector=136,
                             name=Counters.PXDP_TX_128B_DATA))
        # RXC #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=104,
                             name=Counters.RXC_EQ_ALL_SLICES_BUSY,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=180,
                             name=Counters.RXC_CQ_ALL_SLICES_BUSY,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=43,
                             name=Counters.RXC_MSIX_ALL_SLICES_BUSY,
                             units=UnitsMeasure.CYCLES,
                             utilization_reference=Reference.CYCLES,
                             utilization_good=10,
                             utilization_bad=30))
        # This is commeted out for now, due to possible bug in the counter,
        # RM: 1688126.
        #self.counters.append(Counter(
        #                     unit=self.get_unit(Units.RXC),
        #                     selector=98,
        #                     name=Counters.RXC_CQE_ZIP_OPEN_SESSION,
        #                     units=UnitsMeasure.EVENTS,
        #                     utilization_reference=Reference.RX_PACKETS,
        #                     utilization_good=15,
        #                     utilization_bad=50))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=99,
                             name=Counters.RXC_CQE_ZIP_MERGING_CQE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.RX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=109,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_EQE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=113,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_TIMEOUT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=119,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_NOT_MATCH,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=120,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_PX_IDLE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=121,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_S2CQE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=100,
                             name=Counters.RXC_CQE_ZIP_WRITING_8_CQES,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.RXC),
                             selector=122,
                             name=Counters.RXC_CLOSING_ZIP_SESSION_LRO,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.
                             CQE_ZIPPING_SESSIONS,
                             utilization_good=1,
                             utilization_bad=33))
        # SXW #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=279,
                             name=Counters.SXW_QP_DONE_DUE_TO_LIMITED,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=284,
                             name=Counters.SXW_QP_DONE_DUE_TO_VL_LIMITED,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=283,
                             name=Counters.SXW_QP_DONE_DUE_TO_DESCHED,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=10))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=280,
                             name=Counters.SXW_QP_DONE_DUE_TO_WORK_DONE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=10,
                             utilization_bad=33))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=278,
                             name=Counters.SXW_QP_DONE_DUE_TO_E2E_CREDITS,
                             units=UnitsMeasure.EVENTS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXW),
                             selector=31,
                             name=Counters.SXW_PACKET_SEND_SXW2SXP_GO_VID,
                             units=UnitsMeasure.PACKETS))
        # SXP #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=47,
                             name=Counters.SXP_LINE_TRANSMITTED_PORT0,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=46,
                             name=Counters.SXP_LINE_TRANSMITTED_PORT1,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=45,
                             name=Counters.SXP_LINE_TRANSMITTED_LOOPBACK,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=23,
                             name=Counters.SXP_BW_COUNT_PERF_COUNT_0_1))
        """
        The counter below requires configuration
        in order to count TX packet rate.
        This is done by writing 1 to the filtering fields in BW counter below.
        This will count the last transmit of a packet.
        ADB paths:
        .sxp.sxp_bandwidth_counters.transmit_bandwidth_counter.
        bandwidth_counter_filter[1].filter_data.last
        .sxp.sxp_bandwidth_counters.transmit_bandwidth_counter.
        bandwidth_counter_filter[1].filter_mask.last
        """
        self.device.write_field(val=1, addr=0x38fa4, startBit=25, size=1)
        self.device.write_field(val=1, addr=0x38fb4, startBit=25, size=1)
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXP),
                             selector=24,
                             name=Counters.SXP_BW_COUNT_PERF_COUNT_0_2))

        # SXS #
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS),
                             selector=24,
                             name=Counters.
                             SXP_STEERING_PERF_COUNT_STEERING0_RSE_WORK_RATE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS),
                             selector=26,
                             name=Counters.
                             SXP_STEERING_PERF_COUNT_STEERING1_RSE_WORK_RATE,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS),
                             selector=40,
                             name=Counters.SXP_PERF_COUNT_STEERING0_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS),
                             selector=38,
                             name=Counters.SXP_PERF_COUNT_STEERING0_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS),
                             selector=41,
                             name=Counters.SXP_PERF_COUNT_STEERING1_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS),
                             selector=39,
                             name=Counters.SXP_PERF_COUNT_STEERING1_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS),
                             selector=30,
                             name=Counters.
                             SXP_PERF_COUNT_STEERING0_CACHE_1_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS),
                             selector=35,
                             name=Counters.
                             SXP_PERF_COUNT_STEERING1_CACHE_1_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS),
                             selector=29,
                             name=Counters.
                             SXP_PERF_COUNT_STEERING0_CACHE_1_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS),
                             selector=34,
                             name=Counters.
                             SXP_PERF_COUNT_STEERING1_CACHE_1_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS),
                             selector=28,
                             name=Counters.SXP_STEERING_0_TOTAL_CACHE_ACCESS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS),
                             selector=33,
                             name=Counters.SXP_STEERING_1_TOTAL_CACHE_ACCESS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS_E2E_CACHE),
                             selector=15,
                             name=Counters.SXP_E2E_CACHE_TOTAL_LOOKUPS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS_E2E_CACHE),
                             selector=24,
                             name=Counters.SXP_E2E_CACHE_HIT,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS_E2E_CACHE),
                             selector=33,
                             name=Counters.SXP_E2E_CACHE_MISS,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=5,
                             utilization_bad=15))
        self.counters.append(Counter(
                             unit=self.get_unit(Units.SXS_E2E_CACHE),
                             selector=42,
                             name=Counters.SXP_E2E_CACHE_LEARN,
                             units=UnitsMeasure.EVENTS,
                             utilization_reference=Reference.TX_PACKETS,
                             utilization_good=1,
                             utilization_bad=1))

    @staticmethod
    def is_device_id_ok(id):
        if id in DeviceIDs.CONNECTX6:
            return True
        return False

    @staticmethod
    def set_latency_counters(mf):
        """ Set latency counters for sampling.
        """
        # Enable PCIe latency measurements:
        mf.write_field(val=1, addr=0x10d500, startBit=1, size=1)

        # Set flavor to measuring PXTL to response:
        mf.write_field(val=1, addr=0x10d508, startBit=0, size=1)

        # PXTL to response:
        # Reset the counters to NULL:
        mf.write_field(val=1, addr=0x10d900, startBit=0, size=1)
        mf.write_field(val=1, addr=0x10d900, startBit=1, size=1)
        mf.write_field(val=1, addr=0x10d900, startBit=2, size=1)

        # Enables the counter that will be used to measure latency:
        mf.write_field(val=1, addr=0x10d904, startBit=0, size=1)

        # Release counter lock:
        mf.write_field(val=0, addr=0x10d904, startBit=1, size=1)

        # PXTL to PXDP:
        # Reset the counters to NULL:
        mf.write_field(val=1, addr=0x10d700, startBit=0, size=1)
        mf.write_field(val=1, addr=0x10d700, startBit=1, size=1)
        mf.write_field(val=1, addr=0x10d700, startBit=2, size=1)

        # Enables the counter that will be used to measure latency:
        mf.write_field(val=1, addr=0x10d704, startBit=0, size=1)

        # Release counter lock:
        mf.write_field(val=0, addr=0x10d704, startBit=1, size=1)

    def get_latency_counters(self, mf, dev_clock):
        """ Returns latency counters.
        """
        # Reset the counters to NULL:
        mf.write_field(val=0, addr=0x10d900, startBit=0, size=1)
        mf.write_field(val=0, addr=0x10d900, startBit=1, size=1)
        mf.write_field(val=0, addr=0x10d900, startBit=2, size=1)
        mf.write_field(val=0, addr=0x10d700, startBit=0, size=1)
        mf.write_field(val=0, addr=0x10d700, startBit=1, size=1)
        mf.write_field(val=0, addr=0x10d700, startBit=2, size=1)

        # Lock counters for sum/cnt:
        mf.write_field(val=0, addr=0x10d904, startBit=0, size=1)
        mf.write_field(val=1, addr=0x10d904, startBit=1, size=1)
        mf.write_field(val=0, addr=0x10d704, startBit=0, size=1)
        mf.write_field(val=1, addr=0x10d704, startBit=1, size=1)

        # Read latency counters:
        link0_pxtl_to_pcie_response_latency_sum_31_0 = mf.read4(0x010d91c)
        link0_pxtl_to_pcie_response_latency_sum_63_32 = mf.read4(0x010d918)
        link0_pxtl_to_pcie_response_latency_cnt_31_0 = mf.read4(0x010d914)
        link0_pxtl_to_pcie_response_min_latency = mf.read4(0x010d908)
        link0_pxtl_to_pcie_response_max_latency = mf.read4(0x010d90c)
        link0_pxtl_to_pxdp_latency_sum_31_0 = mf.read4(0x010d71c)
        link0_pxtl_to_pxdp_latency_sum_63_32 = mf.read4(0x010d718)
        link0_pxtl_to_pxdp_latency_cnt_31_0 = mf.read4(0x010d714)
        link0_pxtl_to_pxdp_min_latency = mf.read4(0x010d708)
        link0_pxtl_to_pxdp_max_latency = mf.read4(0x010d70c)

        pxtl_to_pcie_response_latency_link0 = 0
        if link0_pxtl_to_pcie_response_latency_cnt_31_0:
            pxtl_to_pcie_response_latency_sum_total = \
                link0_pxtl_to_pcie_response_latency_sum_31_0 + \
                (link0_pxtl_to_pcie_response_latency_sum_63_32 << 32)
            pxtl_to_pcie_response_latency_link0 = \
                pxtl_to_pcie_response_latency_sum_total * 1000 / \
                link0_pxtl_to_pcie_response_latency_cnt_31_0 / dev_clock
        pxtl_to_pcie_response_latency_link0_min = \
            link0_pxtl_to_pcie_response_min_latency * 1000 / dev_clock
        pxtl_to_pcie_response_latency_link0_max = \
            link0_pxtl_to_pcie_response_max_latency * 1000 / dev_clock

        pxtl_to_pxdp_latency_link0 = 0
        if link0_pxtl_to_pxdp_latency_cnt_31_0:
            pxtl_to_pxdp_latency_sum_total = \
                link0_pxtl_to_pxdp_latency_sum_31_0 + \
                (link0_pxtl_to_pxdp_latency_sum_63_32 << 32)
            pxtl_to_pxdp_latency_link0 = \
                pxtl_to_pxdp_latency_sum_total * 1000 / \
                link0_pxtl_to_pxdp_latency_cnt_31_0 / dev_clock
        pxtl_to_pxdp_latency_link0_min = \
            link0_pxtl_to_pxdp_min_latency * 1000 / dev_clock
        pxtl_to_pxdp_latency_link0_max = \
            link0_pxtl_to_pxdp_max_latency * 1000 / dev_clock

        # In case of counter overflow:
        if pxtl_to_pxdp_latency_link0 == 0:
            pxtl_to_pxdp_latency_link0_min = 0
            pxtl_to_pxdp_latency_link0_max = 0

        if pxtl_to_pcie_response_latency_link0 == 0:
            pxtl_to_pcie_response_latency_link0_min = 0
            pxtl_to_pcie_response_latency_link0_max = 0

        # Initialize counters objects:
        self.regular_counters.append(
            Counter(name=Counters.PXT_TO_PXDP_LATENCY_LINK0_AVG,
                    value=pxtl_to_pxdp_latency_link0))
        self.regular_counters.append(
            Counter(name=Counters.PXT_TO_PXDP_LATENCY_LINK0_MIN,
                    value=pxtl_to_pxdp_latency_link0_min))
        self.regular_counters.append(
            Counter(name=Counters.PXT_TO_PXDP_LATENCY_LINK0_MAX,
                    value=pxtl_to_pxdp_latency_link0_max))
        self.regular_counters.append(
            Counter(name=Counters.PCI_LATENCY_LINK0_AVG,
                    value=(pxtl_to_pcie_response_latency_link0 -
                           pxtl_to_pxdp_latency_link0)))
        self.regular_counters.append(
            Counter(name=Counters.PCI_LATENCY_LINK0_MIN,
                    value=(pxtl_to_pcie_response_latency_link0_min -
                           pxtl_to_pxdp_latency_link0_min)))
        self.regular_counters.append(
            Counter(name=Counters.PCI_LATENCY_LINK0_MAX,
                    value=(pxtl_to_pcie_response_latency_link0_max -
                           pxtl_to_pxdp_latency_link0_max)))
