# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Simon Raviv
# @date: April 02, 2017


class LatencyBreakDownSelector:
    """
    """
    def __init__(self, name, selector_value):
        self.name = name
        self.selector_value = selector_value
        self.average = 0
        self.minimum = 0
        self.maximum = 0


class LatencyBreakDownUnit:
    """
    """
    def __init__(self, name, en_addr, reset_addr,
                 stopped_max_samples_addr, total_latency_addr,
                 total_events_addr, min_addr, max_addr,
                 selector_addr, selector_size, selector_offset=0):
        self.name = name
        self.en_addr = en_addr
        self.reset_addr = reset_addr
        self.stopped_max_samples_addr = stopped_max_samples_addr
        self.total_latency_addr = total_latency_addr
        self.total_events_addr = total_events_addr
        self.min_addr = min_addr
        self.max_addr = max_addr
        self.selector_addr = selector_addr
        self.selector_size = selector_size
        self.selector_offset = selector_offset
        self.selectors = []
        self.reset_unit()

    def enable(self, mf):
        """
        """
        mf.write_field(val=1, addr=self.en_addr, startBit=0, size=1)

    def disable(self, mf):
        """
        """
        mf.write_field(val=0, addr=self.en_addr, startBit=0, size=1)

    def reset_values(self, mf):
        """
        """
        mf.write_field(val=1, addr=self.reset_addr, startBit=31, size=1)

    def reset_unit(self):
        """
        """
        self.current_selector_index = -1
        self.is_done = False

    def set_selector(self, mf, selector_index):
        """
        """
        if self.selector_addr is not None \
                and self.selector_offset is not None \
                and self.selector_size is not None:
            mf.write_field(val=self.selectors[selector_index].selector_value,
                           addr=self.selector_addr,
                           startBit=self.selector_offset,
                           size=self.selector_size)

    def sample(self, mf):
        """
        """
        selector_to_sample = self.selectors[self.current_selector_index]
        selector_to_sample.average = self.get_average(mf)
        selector_to_sample.minimum = self.get_minimum(mf)
        selector_to_sample.maximum = self.get_maximum(mf)

    def is_stopped_max_samples(self, mf):
        """
        """
        return mf.read_field(addr=self.stopped_max_samples_addr,
                             startBit=1, size=1) == 1

    def get_average(self, mf):
        """
        """
        total_latency = mf.read_field(addr=self.total_latency_addr,
                                      startBit=0, size=24)
        total_events = mf.read_field(addr=self.total_events_addr,
                                     startBit=0, size=16)
        if total_events > 0:
            return total_latency / total_events
        return 0

    def get_minimum(self, mf):
        """
        """
        return mf.read_field(addr=self.min_addr, startBit=0, size=16)

    def get_maximum(self, mf):
        """
        """
        return mf.read_field(addr=self.max_addr, startBit=16, size=16)

    def add_selector(self, name, selector_value):
        """
        """
        self.selectors.append(LatencyBreakDownSelector(name, selector_value))

    def set_next_selector(self, mf):
        """
        """
        if self.current_selector_index == (len(self.selectors) - 1):
            self.is_done = True
            return self.is_done

        self.current_selector_index += 1
        self.set_selector(mf, self.current_selector_index)
        return self.is_done

    def get_selector_name(self):
        """
        """
        return self.selectors[self.current_selector_index].name

    def get_done_percentage(self):
        """
        """
        if self.is_done:
            return 100
        else:
            return 100 * \
                   (self.current_selector_index + 1) /\
                   len(self.selectors)


class MCRACounters(object):
    """ Represents MCRA counters.
    """
    def __init__(self, device=None):
        self.units = []
        self.latency_break_down_unit_list = []
        self.counters = []
        self.regular_counters = []
        self.device = device
        self._pci_link_gen_address = None

    @staticmethod
    def is_device_id_ok(id):
        """ Check if the device ID is valid.
        """
        return False

    def get_unit(self, name):
        """ Returns counter's unit.
        """
        x = next(x for x in self.units if x.name == name)
        return x

    def get_lbd_unit(self, name):
        """ Returns counter's LBD unit.
        """
        return filter(lambda unit: unit.name == name,
                      self.latency_break_down_unit_list)[0]

    def get_counters_per_unit(self):
        """ Returns grouping by counters per chip's unit.
        """
        result = dict()

        for unit in self.units:
            result[unit.name] = list()

        for counter in self.counters:
            unit = counter.get_unit_name()
            result[unit].append(counter.name)

        return result

    def _get_pci_link_gen(self):
        """ Return PCI link generation.
        """
        pci_link_gen = self.device.read_field(
            addr=self._pci_link_gen_address,
            startBit=20,
            size=3)
        return pci_link_gen


class MCRAValue(object):
    """ Represents MCRA value.
    """

    def __init__(self, mf, address):
        self.address = address
        try:
            self.value = mf.read4(address)
        except:
            self.value = 0

    def __int__(self):
        return self.value
