# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Simon Raviv
# @date: April 02, 2017

import os
import platform
import ctypes
from performance.performance_exceptions import PerformanceException


def ones(n):
    """
    """
    return (1 << n) - 1


def extract_field(val, start, size):
    """
    """
    return (val & (ones(size) << start)) >> start


def insert_field(val1, start1, val2, start2, size):
    """
    """
    return val1 & ~(ones(size) << start1) | \
        (extract_field(val2, start2, size) << start1)


class MtcrException(PerformanceException):
    """ Represents MTCR exception.
    """
    def __init__(self, msg_str, error_code=1):
        super(MtcrException, self).__init__(msg_str, error_code)


CMTCR = None
try:
    from ctypes import *

    if platform.system() == "Windows" or os.name == "nt":
        CMTCR = CDLL("libmtcr-1.dll", use_errno=True)
    else:
        try:
            CMTCR = CDLL("cmtcr.so", use_errno=True)
        except:
            CMTCR = CDLL(os.path.join(
                        os.path.dirname(os.path.realpath(__file__)),
                        "cmtcr.so"),
                        use_errno=True)
except:
    CMTCR = None

if CMTCR:
    class MstDevice(object):
        """
        """

        def __init__(self, dev):
            self.mf = 0
            self.mopenFunc = CMTCR.mopen
            self.mopenFunc.restype = c_void_p
            self.mcloseFunc = CMTCR.mclose
            self.mread4Func = CMTCR.mread4
            self.mwrite4Func = CMTCR.mwrite4
            self.mread_4blockFunc = CMTCR.mread4_block
            self.mwrite_4blockFunc = CMTCR.mwrite4_block
            self.icmdSendCommandFunc = CMTCR.icmd_send_command
            self.m_hca_resetFunc = CMTCR.mhca_reset

            self.mf = ctypes.c_void_p(self.mopenFunc(dev))
            if not self.mf:
                raise MtcrException("Failed to open device (%s): %s" %
                                    (dev, os.strerror(ctypes.get_errno())))

        def close(self):
            """
            """
            if self.mf:
                self.mcloseFunc(self.mf)
            self.mf = None

        def __del__(self):
            """
            """
            self.close()

        def read4(self, addr):
            """
            """
            val = c_uint32()
            if self.mread4Func(self.mf, addr, byref(val)) != 4:
                raise MtcrException(
                    "Failed to read from mst device from address: 0x%x"
                    % addr)

            return val.value

        def read_field(self, addr, startBit, size):
            """
            """
            return extract_field(self.read4(addr), startBit, size)

        def write4(self, addr, val):
            """
            """
            cval = c_uint32(val)
            if self.mwrite4Func(self.mf, addr, cval) != 4:
                raise MtcrException(
                    "Failed to write to mst device to address: 0x%x" % addr)

        def write_field(self, val, addr, startBit, size):
            """
            """
            oldVal = self.read4(addr)
            newVal = insert_field(oldVal, startBit, val, 0, size)

            self.write4(addr, newVal)

        def read_4block(self, addr, size):  # size in dwords
            """
            """
            data_arr = (c_uint32 * size)()
            if self.mread_4blockFunc(self.mf,
                                     addr,
                                     cast(data_arr, POINTER(c_uint32)),
                                     size * 4) != size * 4:
                raise MtcrException(
                    "Failed to read block from mst device from address\
                    0x%x of size %d bytes" % (addr, size))
            return list(data_arr)

        def write_4block(self, addr, data_list):
            """
            """
            size = len(data_list)
            data_arr = (c_uint32 * size)(*data_list)
            if self.mwrite_4blockFunc(self.mf,
                                      addr,
                                      cast(data_arr, POINTER(c_uint32)),
                                      size * 4) != size * 4:
                raise MtcrException(
                    "Failed to write block to mst device to address\
                    0x%x of size %d bytes" % (addr, size))

        def icmd_send_cmd(self, opcode, data, skip_write):
            """
            """
            data_arr = (c_uint8 * len(data))(*data)
            rc = self.icmdSendCommandFunc(self.mf,
                                          opcode,
                                          cast(data_arr, POINTER(c_uint8)),
                                          len(data),
                                          skip_write)
            if rc:
                raise MtcrException("Failed to send command")

        def m_hca_reset(self):
            """
            """
            if self.m_hca_resetFunc(self.mf) != 0:
                raise MtcrException("Failed to reset device")
elif False:
    import subprocess
    import string

    def get_status_output(cmd):
        """
        """
        pipe = subprocess.Popen(cmd,
                                shell=True,
                                stdout=subprocess.PIPE,
                                stderr=subprocess.STDOUT)
        cout = pipe.stdout
        output = cout.read()
        cout.close()
        rc = pipe.wait()
        return rc, output

    class MstDevice(object):
        """
        """

        def __init__(self, dev):
            self.dev = dev
            self.mf = 0

        def read4(self, addr):
            """
            """
            cmd = "mcra %s 0x%x" % (self.dev, addr)
            rc, out = get_status_output(cmd)
            if rc:
                raise MtcrException(
                    "Failed to read from mst device from address 0x%x: %s" %
                    (addr, out))
            return string.atoi(out, 16)

        def read_field(self, addr, startBit, size):
            """
            """
            return extract_field(self.read4(addr), startBit, size)

        def write4(self, addr, val):
            """
            """
            cmd = "mcra %s 0x%x 0x%x" % (self.dev, addr, val)
            rc, out = get_status_output(cmd)
            if rc:
                raise MtcrException(
                    "Failed to wrtie to mst device to address 0x%x: %s" %
                    (addr, out))

        def write_field(self, val, addr, startBit, size):
            """
            """
            oldVal = self.read4(addr)
            newVal = insert_field(oldVal, startBit, val, 0, size)
            self.write4(addr, newVal)

        def read_4block(self, addr, size):
            """
            """
            read_list = []
            for add in range(addr, addr + size * 4, 4):
                read_list.append(self.read4(add))
            return read_list

        def write_4block(self, addr, data_list):
            """
            """
            size = len(data_list)
            for i, add in enumerate(range(addr, addr + size * 4, 4)):
                self.write4(add, data_list[i])

        def icmd_send_cmd(self, opcode, data, skip_write):
            """
            """
            raise MtcrException("icmd isn't supported in MCRA mode")

        def m_hca_reset(self):
            """
            """
            raise MtcrException("mswReset isn't supported in MCRA mode")
else:
    raise MtcrException("Failed to load cmtcr.so")
