# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --


# @author: Simon Raviv
# @date: April 02, 2017


import abc
import signal
from Queue import Empty
from multiprocessing import Process, Queue
from performance.services.communication.alert import Alert
from performance.services.communication.alert_code import AlertCode
from performance.services.communication.cmd import Cmd
from performance.services.communication.op_code import OpCode
from performance.entities.data import Data
from performance.services.logger import PerformanceLog


class Worker(object):
    """
    """
    __metaclass__ = abc.ABCMeta

    def __init__(self, name=None, max_timeout=20, is_external=False):
        self.name = name
        self.data = None
        self.process = None
        self.parent_worker = None
        self.children_workers = []
        self.had_children = False
        self.children_done = 0
        self.error_str = None
        self.max_iterations = None
        self.iteration = 0
        self.Q = Queue()
        self.timeout = max_timeout  # In seconds
        self.is_external = is_external
        self.log = PerformanceLog().get_log()

    @abc.abstractmethod
    def main_func(self):
        """
        """
        return

    def get_data(self):
        """ Returns software data structure.
        """
        return self.data

    def check_iterations(self):
        """
        """
        if self.max_iterations is not None:
            if self.iteration == self.max_iterations:
                return True
        return False

    def add_child(self, child):
        """
        """
        self.had_children = True
        self.children_workers.append(child)

    def remove_broken_children(self):
        """
        """
        working_children = []
        for cw in self.children_workers:
            cw.remove_broken_children()
            if cw.can_work():
                working_children.append(cw)
        self.children_workers = working_children

    def can_work(self):
        """
        """
        if self.error_str is None:
            no_error = True
        is_leaf = (self.children_workers == [] and not self.had_children)
        any_working_children = any(
            [cw for cw in self.children_workers if cw.can_work()])
        return (no_error and (is_leaf or any_working_children))

    def main_loop(self):
        """
        """
        exit = False
        obj = None
        # Set worker thread to ignore Ctrl+C interrupt.
        signal.signal(signal.SIGINT, signal.SIG_IGN)
        while not exit:
            try:
                obj = self.Q.get(timeout=self.timeout)
            except Empty:
                self.log.error("Worker reached timeout")
                exit = True

            if isinstance(obj, Data):
                self.data = obj
                self.main_func()
                if self.parent_worker:
                    self.parent_worker.Q.put(self.data)

            if isinstance(obj, Cmd):
                cmd = obj
                for c_worker in self.children_workers:
                    c_worker.Q.put(cmd)
                if cmd.op_code == OpCode.Stop:
                    exit = True

            if isinstance(obj, Alert):
                alert = obj
                if alert.code == AlertCode.FINISHED:
                    self.children_done += 1
                exit = (self.children_done == len(self.children_workers))

        if self.parent_worker:
            self.parent_worker.Q.put(Alert(AlertCode.FINISHED))

    def get_error(self):
        """
        """
        return self.error_str

    def set_error(self, err):
        """
        """
        if err:
            if self.error_str:
                self.error_str += '\n' + err
            else:
                self.error_str = err

    def start(self, max_iterations=None):
        """
        """
        self.max_iterations = max_iterations
        self.process = Process(target=self.main_loop)
        self.process.start()
        for cw in self.children_workers:
            cw.start(max_iterations)

    def is_alive(self):
        """
        """
        return self.process.is_alive()
