#!/usr/bin/env python2

import os
import sys
import json
import time
import argparse


class CKAnalysis:

    def __init__(self, filename):
        self.filename = filename
        self.vals = {}
        self.load()

    def load(self):
        try:
            with open(self.filename) as json_data:
                d = json.load(json_data)
                json_data.close()
                self.json = d
                self.valid = True
            return d
        except Exception as e:
            print "Failed to parse json file: ", self.filename
            self.valid = False

    def value(self, x, y, links ):
        if x == y:
            return 0.0
        if x > y:
            x, y = y, x

        return links[x][y - x]


    def get_data(self, n):
        if n not in self.vals:
            self.vals[n] = {}
        return self.vals[n]

    def save(self, n1, n2, v):
        d1 = self.get_data(n1)
        d1[n2] = v

        d2 = self.get_data(n2)
        d2[n1] = v

    def process(self, threshold, cmp):

        thresh = float(threshold)
        print "Threshold: ", thresh

        nodes = self.json["Nodes"]
        links = self.json["Links"]
        num_items = len(nodes)

        nodes_by_idx = {}
        for k in nodes.keys():
            nodes_by_idx[nodes[k]] = k

        nodes = nodes_by_idx

        t0 = time.time()

        gt = "gt" in cmp or "above" in cmp
        lt = "lt" in cmp or "below" in cmp
        for i in xrange(num_items):
            n1 = nodes[i]
            for j in xrange(i):
                n2 = nodes[j]
                val = self.value(i, j, links)
                if gt and val > thresh:
                    self.save(n1, n2, val)
                if lt and val < thresh:
                    self.save(n1, n2, val)

        t1 = time.time()
        #print "Processing took %f sec" % (t1 - t0)


if __name__ == "__main__":

    def get_compare(args):
        if args.compare is None:
            filename = args.filename

            if "latency" in filename:
                return "gt"
            if "bandwidth" in filename:
                return "lt"
        else:
            return args.compare

    def get_threshold(args):
        if args.threshold is None:
            filename = args.filename
            if "latency" in filename:
                return 2.15

            if "bandwidth" in filename:
                return 23000
        else:
            return args.threshold


    def parse_args():
        prog_name = os.path.basename(__file__)
        desc = 'bandwidth/latency report generator.  Shows all pairs with bw/latency [above/below] a given threshold'
        parser = argparse.ArgumentParser(prog=prog_name, description=desc)
        parser.add_argument('-f', '--filename', required=True, help='.json file to be processed')
        parser.add_argument('-t', '--threshold', required=False, help='selection threshold')
        parser.add_argument('-c', '--compare', required=False, default=None, help='Comparison [above|gt||below|lt]')
        parser.add_argument('-n', '--num_poor', required=False, default=1, help='Number of poor links a given node has to participate in to be included in the report')
        return parser.parse_args()

    args = parse_args()
    f = args.filename

    if os.path.exists(f):

        compare = get_compare(args)
        threshold = get_threshold(args)
        num_poor = int(args.num_poor)

        p = CKAnalysis(f)
        p.process(threshold, compare)

        print "From   To"
        for k in sorted(p.vals):
            if len(p.vals[k]) >= num_poor:
                print k,
                for e in sorted(p.vals[k]):
                    print "(%s : %3.2f)," % (e, p.vals[k][e]),
                print
    else:
        print "File ", args.filename, " not found"
        sys.exit(-1)
