#!/usr/bin/env python

# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --
# Author: Ahmed Awwad   ahmadaw@mellanox.com -- created: 2017/March

# Python Imports
import os
import re
import json
import time
import random
import optparse
# local imports
from neohost_socket_server import NBESocketServer
from neohost_shell_server import NBEShellServer
import neohost_sdk_constants as NC
from neohost_sdk_exception import NeohostSdkException
from print_data import PrintData


class CommandSdk(object):
    """
    This class for command sdk common module
    """
    MAX_ID_NUM = 1000
    MIN_ID_NUM = 1

    def __init__(self, timeout=500):
        """
        Constructor for command_sdk
        """

        self._cmdParser      = optparse.OptionParser()
        self.__timeout       = timeout
        self._options        = None
        self._args           = None
        self.__port          = None
        self._printObj       = None
        self._execMode       = None
        self.__runMode       = None
        self.__withDebug     = None
        self.__viaSocket     = None
        self.__execModeStr   = None
        self.__neohostServer = None
        self.__neohostClient = None
        self.request         = {}
        self._returnRespond   = False
        self._serverProcess = None
        self._outputFormat = None
        self._outputFormatDefault = NC.OutputFormatOptions.JSON
        self._outputFormatOptions = [NC.OutputFormatOptions.JSON]

    def addOptions(self):
        """
        :param parser: add options
        """
        self._cmdParser.add_option("--exec-mode",
                                  dest="execMode",
                                  help="Execution mode for neohost [sync, a-sync]",
                                  default=NC.STR_SYNC)
        self._cmdParser.add_option("--mode", dest="runMode",
                                  help="Running mode for neohost Shell/Socket",
                                  default=None)
        self._cmdParser.add_option("--port",
                                  dest="socketPort",
                                  help="Running port for neohost socket",
                                  default=None)
        self._cmdParser.add_option("--DEBUG",
                                  dest="withDebug",
                                  action="store_true",
                                  help="Running Sdk in DEBUG mode",
                                  default=False)
        self._cmdParser.add_option("--output-format",
                                   dest="outputFormat",
                                   help=("Output text options, "
                                         "available options: {options},"
                                         " default: {default}").
                                         format(options=self._outputFormatOptions,
                                                default=self._outputFormatDefault),
                                   choices=self._outputFormatOptions,
                                   default=NC.OutputFormatOptions.JSON)

    def parseOptions(self):
        """
        parse options
        """
        # init options and args as cmdParser --sync --async
        (self._options, self._args) = self._cmdParser.parse_args()
        self.__port = self._options.socketPort
        self.__execModeStr = self._options.execMode
        self.__runMode = self._options.runMode
        self.__withDebug = self._options.withDebug
        self._execMode = (NC.SYNCHRONOUS, NC.A_SYNCHRONOUS)[self.__execModeStr == NC.STR_A_SYNC]
        self.__viaSocket = (False, True)[self.__runMode == NC.SOCKET_MODE]
        self._printObj = PrintData(withDebug=self.__withDebug)

    def validateOptions(self):
        """
        This is to validate command options
        """
        if self.__runMode not in NC.NEOHOST_MODES:
            raise NeohostSdkException("Invalid option for --mode. Please choose from %s." % NC.NEOHOST_MODES)
        if self.__execModeStr not in NC.NEOHOST_EXEC_MODES:
            raise NeohostSdkException("Invalid option for --exec-mode. Please choose from %s." % NC.NEOHOST_EXEC_MODES)
        if self.__execModeStr == NC.STR_A_SYNC:
            raise NeohostSdkException("Not Supported option: '%s'. Please use '%s' mode until it's supported." % (NC.STR_A_SYNC, NC.STR_SYNC))
        if self.__runMode == "socket" and not self.__port:
            raise NeohostSdkException("Missing option --port.")
        if not self.__execModeStr:
            raise NeohostSdkException("Missing option. No Execution mode was selected. Please use --exec-mode sync or a-sync.")

    def prepareRequest(self):
        """
        This is to prepare a  request before post it
        """
        # Use the Constants
        self.request = {NC.ATTR_ID: self.__generateRandomId(),
                        NC.ATTR_API_VER: NC.VAL_API_VER,
                        NC.ATTR_PARAMS: None,
                        NC.ATTR_EXEC_MODE: self._execMode
                        }

    def postRequest(self):
        """
        This is to post a request to neohost
        :return:
        """
        self._printObj.printOutput("-I- Post request %s to neohost" % self.request)
        if self.__viaSocket:
            self.__neohostServer = NBESocketServer(NC.NEOHOST_COMMAND, socketPort=self.__port)
            self.__neohostClient = self.__neohostServer.createClientObject()
        else:
            self.__neohostServer = NBEShellServer(NC.NEOHOST_COMMAND)
            self._serverProcess = self.__neohostServer.start(self._serverProcess)
        self.__neohostServer.postRequest(self.request, self.__neohostClient)

    def getResponse(self):
        """
        This is to get a response from neohost
        :return:
        """
        response = self.__neohostServer.getResponse(clientObj=self.__neohostClient, timeOut=self.__timeout)
        if not self._returnRespond:
            print(response)
        # Removal of terminal problematic characters: 
        ansiEscape = re.compile(r'(\x9B|\x1B\[)[0-?]*[ -/]*[@-~]')
        jsonResponse = ansiEscape.sub('', response)
        return jsonResponse

    def __generateRandomId(self):
        """
        This is to generate random id for a request
        :return:
        """
        idNum = random.randint(self.MIN_ID_NUM, self.MAX_ID_NUM)
        return idNum

    def runCommand(self):
        """
        This is to run the command
        :return:
        """
        rc = NC.RC_SUCCESS
        self.addOptions()
        self.parseOptions()
        self.validateOptions()
        self.prepareRequest()
        self.postRequest()
        response = self.getResponse()
        if "error" in json.loads(response).keys():
            rc = NC.RC_FAIL
        self.__neohostServer.stop()
        return rc

    @classmethod
    def main(cls):
        """
        Main for all the commands
        :return:
        """
        rc = NC.RC_SUCCESS
        try:
            command_obj = cls()
            rc = command_obj.runCommand()
        except Exception as e:
            print("-E- %s" % e)
            rc = NC.RC_FAIL
        return rc
