#!/usr/bin/env python

# --
#                 - Mellanox Confidential and Proprietary -
#
# Copyright (C) Jan 2013, Mellanox Technologies Ltd.  ALL RIGHTS RESERVED.
#
# Except as specifically permitted herein, no portion of the information,
# including but not limited to object code and source code, may be reproduced,
# modified, distributed, republished or otherwise exploited in any form or by
# any means for any purpose without the prior written permission of Mellanox
# Technologies Ltd. Use of software subject to the terms and conditions
# detailed in the file "LICENSE.txt".
# --
# Author: Ahmed Awwad   ahmadaw@mellanox.com -- created: 2017/March

# Python Imports
import os
import sys
# local imports
from command_sdk import CommandSdk
import neohost_sdk_constants as NC
from neohost_sdk_exception import NeohostSdkException


class GetDeviceFWImageInfo(CommandSdk):
    """
    This class to get device fw image info
    """

    def __init__(self):
        """
        Constructor for GetDeviceFWImageInfo
        """
        super(GetDeviceFWImageInfo, self).__init__()
        self.__devUid = None
        self.__devPSID = None
        self.__filePath = None
        self.__isMFAImage = False

    def addOptions(self):
        """
        add options
        """
        super(GetDeviceFWImageInfo, self).addOptions()
        self._cmdParser.add_option("--file-path", dest="filePath",
                                   help=NC.HELP_FILE_PATH,
                                   default=None)
        self._cmdParser.add_option("--dev-uid", dest="devUid",
                                   help=NC.HELP_DEV_UID,
                                   default=None)
        self._cmdParser.add_option("--dev-psid", dest="devPSID",
                                   help=NC.HELP_DEV_PSID,
                                   default=None)
        self._cmdParser.add_option("--attr-mask", dest="attrMask",
                                   help=NC.HELP_ATTR_MASK,
                                   type=int,
                                   default=0)

    def parseOptions(self):
        """
        parse options
        """
        super(GetDeviceFWImageInfo, self).parseOptions()
        self.__devUid = self._options.devUid
        self.__devPSID = self._options.devPSID
        self.__filePath = self._options.filePath
        self.__isMFAImage = self.__isMFAImageCheck()
        self.__attrMask = self._options.attrMask

    def prepareRequest(self):
        """
        Prepare request before post it . add method and module.
        """
        super(GetDeviceFWImageInfo, self).prepareRequest()
        self.request[NC.ATTR_METHOD] = NC.METHOD_GET_DEVICE_FW_IMAGE_INFO
        self.request[NC.ATTR_MODULE] = NC.MODULE_MFT_FW
        self.request[NC.ATTR_PARAMS] = {NC.PARAM_FILE_PATH: self.__filePath,
                                        NC.PARAM_ATTR_MASK: self.__attrMask}
        if self.__isMFAImage:
            if self.__devUid:
                self.request[NC.ATTR_PARAMS][NC.PARAM_DEV_UID] = self.__devUid
            if self.__devPSID:
                self.request[NC.ATTR_PARAMS][NC.PARAM_DEV_PSID] = self.__devPSID

    def validateOptions(self):
        """
        Validate Options
        """
        super(GetDeviceFWImageInfo, self).validateOptions()
        if not self.__filePath:
            raise NeohostSdkException("-E- %s" % NC.MISSING_FILE_PATH)
        # If devUID or devPSID is given for not MFA Image
        if (self.__devPSID or self.__devUid) and not self.__isMFAImage:
            raise NeohostSdkException(NC.WRONG_PARAMETER_CHOICE_MFA)
        # If devUID and devPSID are not given for MFA Image
        if self.__devPSID is None and self.__devUid is None and self.__isMFAImage:
            raise NeohostSdkException(NC.MISSING_PARAMETER_CHOICE_MFA)

    def __isMFAImageCheck(self):
        """
        This is to check if the image is MFA (return True) else return False
        By checking the suffix of the image
        """
        isMFAImage = False
        if self.__filePath and self.__filePath.endswith(".mfa"):
            isMFAImage = True
        return isMFAImage


if __name__ == "__main__":
    rc = GetDeviceFWImageInfo.main()
    sys.exit(rc)
