# Copyright (c) Sep 2019 Mellanox Technologies LTD. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --

from __future__ import print_function
import sys
import os
import string
import subprocess
import re
import platform
import regaccess
import time
from enum import Enum

class DeviceType(Enum):
    DEVICE_CONNECTIB = 0
    DEVICE_SWITCHIB = 1
    DEVICE_CONNECTX4 = 2
    DEVICE_SPECTRUM = 3
    DEVICE_CONNECTX4LX = 4
    DEVICE_SWITCHIB2 = 5
    DEVICE_CONNECTX5 = 6
    DEVICE_QUANTUM = 7
    DEVICE_SPECTRUM2 = 8
    DEVICE_BLUEFIELD = 9
    DEVICE_CONNECTX6 = 10
    DEVICE_CONNECTX6DX = 11
    DEVICE_BLUEFIELD2 = 12
    DEVICE_CONNECTX6LX = 13
    DEVICE_SPECTRUM3 = 14
    DEVICE_CONNECTX7 = 15
    DEVICE_QUANTUM2 = 16
    DEVICE_BLUEFIELD3 = 17
    DEVICE_SPECTRUM4 = 18
    DEVICE_CONNECTX8 = 19
    DEVICE_BLUEFIELD4 = 20
    DEVICE_QUANTUM3 = 21
    DEVICE_GB100 = 22

class FwTraceUtilities(object):

    HCA_MASK_CLASSES = [
        ("DEBUG_INIT", 0), ("INIT", 1), ("ICM", 2), ("ICM_FREE_LIST", 3),
        ("HOST_MNG", 4), ("CMD_IF", 5), ("PHY_IB", 6), ("PHY_RX_ADAP", 7),
        ("LIBFHI", 8), ("PHY_COMMON", 9), ("PHY_MANAGER", 10),
        ("PWR", 11), ("FLR", 12), ("ICM_ACCESS", 13),
        ("MAD", 14), ("RXT_CHECKS", 15), ("I2C", 16), ("TRANSPORT", 17),
        ("FW_LL", 18), ("RX_ERRORS", 19), ("CMD_DRIVER", 20), ("PROFILING", 21),
        ("MANAGEMENT", 22), ("FLASH", 23), ("STEERING", 24),
        ("IFARM", 25), ("ICMD", 26), ("PCI", 27), ("DC_CLEANUP", 28),
        ("PHY_ETH", 29), ("VIRT", 30)]  # list of (trace type name, start_bit)

    #"chip_rev":-1, for all devices consider removing
    #"embedded_fw_str_db": True, for all devices consider removing
    DEV_INFO_DB = [
        # we use list instead of dict to keep order,
        # new devices should be at head of the list
        {
            "name": "ConnectIB",
            "type": DeviceType.DEVICE_CONNECTIB,
            "dev_id": [0x1ff],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": True,
            "fw_str_db_signature_addr": (0x24780, 0, 16),
            "maxMainIrisc" : 8,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xe001c,
            "IriscStep" : 0x200,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": list(HCA_MASK_CLASSES),
            "default_tracer_mode": "MEM",
            "is_hca": True
        },
        {
            "name": "SwitchIB",
            "type": DeviceType.DEVICE_SWITCHIB,
            "dev_id": [0x247],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0xffffffff, 0, 16),
            "maxMainIrisc" : 4,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xa001c,
            "IriscStep" : 0x200,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": [("class1", 0), ("class2", 1)],
            "default_tracer_mode": "FIFO",
            "is_hca": False
        },
        {
            "name": "ConnectX4",
            "type": DeviceType.DEVICE_CONNECTX4,
            "dev_id": [0x209],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0x24780, 0, 16),
            "maxMainIrisc" : 8,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xe001c,
            "IriscStep" : 0x400,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": list(HCA_MASK_CLASSES),
            "default_tracer_mode": "MEM",
            "is_hca": True
        },
        {
            "name": "ConnectX5",
            "type": DeviceType.DEVICE_CONNECTX5,
            "dev_id": [0x20d],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0x24780, 0, 16),
            "maxMainIrisc" : 11,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xe001c,
            "IriscStep" : 0x400,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": list(HCA_MASK_CLASSES),
            "default_tracer_mode": "MEM",
            "is_hca": True
        },
        {
            "name": "BlueField",
            "type": DeviceType.DEVICE_BLUEFIELD,
            "dev_id": [0x211],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0x24780, 0, 16),
            "maxMainIrisc" : 11,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xe001c,
            "IriscStep" : 0x400,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": list(HCA_MASK_CLASSES),
            "default_tracer_mode": "MEM",
            "is_hca": True
        },
        {
            "name": "BlueField2",
            "type": DeviceType.DEVICE_BLUEFIELD2,
            "dev_id": [0x214],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0x24780, 0, 16),
            "maxMainIrisc" : 11,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xe001c,
            "IriscStep" : 0x400,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": list(HCA_MASK_CLASSES),
            "default_tracer_mode": "MEM",
            "is_hca": True
        },
        {
            "name": "BlueField3",
            "type": DeviceType.DEVICE_BLUEFIELD3,
            "dev_id": [0x21c],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0x24780, 0, 16),
            "maxMainIrisc" : 11,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xe001c,
            "IriscStep" : 0x400,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": list(HCA_MASK_CLASSES),
            "default_tracer_mode": "MEM",
            "is_hca": True
        },
        {
            "name": "BlueField4",
            "type": DeviceType.DEVICE_BLUEFIELD4,
            "dev_id": [0x220],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0x24780, 0, 16),
            "maxMainIrisc" : 11,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xe001c,
            "IriscStep" : 0x400,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": list(HCA_MASK_CLASSES),
            "default_tracer_mode": "MEM",
            "is_hca": True
        },
        {
            "name": "ConnectX6",
            "type": DeviceType.DEVICE_CONNECTX6,
            "dev_id": [0x20f],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0x24780, 0, 16),
            "maxMainIrisc" : 11,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xe001c,
            "IriscStep" : 0x400,
            "TileStart" : 0x0,
            "TileStep" : 0x0,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": list(HCA_MASK_CLASSES),
            "default_tracer_mode": "MEM",
            "is_hca": True
        },
        {
            "name": "ConnectX6DX",
            "type": DeviceType.DEVICE_CONNECTX6DX,
            "dev_id": [0x212],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0x24780, 0, 16),
            "maxMainIrisc" : 11,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xe001c,
            "IriscStep" : 0x400,
            "TileStart" : 0x0,
            "TileStep" : 0x0,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": list(HCA_MASK_CLASSES),
            "default_tracer_mode": "MEM",
            "is_hca": True
        },
        {
            "name": "Spectrum",
            "type": DeviceType.DEVICE_SPECTRUM,
            "dev_id": [0x249],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0xffffffff, 0, 16),
            "maxMainIrisc" : 8,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xa001c,
            "IriscStep" : 0x200,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": [("class1", 0), ("class2", 1)],
            "default_tracer_mode": "FIFO",
            "is_hca": True
        },
        {
            "name": "ConnectX4LX",
            "type": DeviceType.DEVICE_CONNECTX4LX,
            "dev_id": [0x20b],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0x24780, 0, 16),
            "maxMainIrisc" : 8,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xe001c,
            "IriscStep" : 0x400,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": list(HCA_MASK_CLASSES),
            "default_tracer_mode": "MEM",
            "is_hca": True
        },
        {
            "name": "SwitchIB2",
            "type": DeviceType.DEVICE_SWITCHIB2,
            "dev_id": [0x24B],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0xffffffff, 0, 16),
            "maxMainIrisc" : 4,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xa001c,
            "IriscStep" : 0x200,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": [("class1", 0), ("class2", 1)],
            "default_tracer_mode": "FIFO",
            "is_hca": False
        },
        {
            "name": "Quantum",
            "type": DeviceType.DEVICE_QUANTUM,
            "dev_id": [0x24D],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0xffffffff, 0, 16),
            "maxMainIrisc" : 8,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xa001c,
            "IriscStep" : 0x200,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": [("class1", 0), ("class2", 1)],
            "default_tracer_mode": "FIFO",
            "is_hca": False
        },
        {
            "name": "Spectrum2",
            "type": DeviceType.DEVICE_SPECTRUM2,
            "dev_id": [0x24E],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0xffffffff, 0, 16),
            "maxMainIrisc" : 8,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xa001c,
            "IriscStep" : 0x200,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": [("class1", 0), ("class2", 1)],
            "default_tracer_mode": "FIFO",
            "is_hca": False
        },
        {
            "name": "Spectrum3",
            "type": DeviceType.DEVICE_SPECTRUM3,
            "dev_id": [0x250],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0xffffffff, 0, 16),
            "maxMainIrisc" : 10,
            "maxAPU" : 0,
            "maxNumOfTiles" : 8,
            "maxIriscPerTile" : 5,
            "IriscStartAddr" : 0xa001c,
            "IriscStep" : 0x200,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": [("class1", 0), ("class2", 1)],
            "default_tracer_mode": "FIFO",
            "is_hca": False
        },
        {
            "name": "Spectrum4",
            "type": DeviceType.DEVICE_SPECTRUM4,
            "dev_id": [0x254],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0xffffffff, 0, 16),
            "maxMainIrisc" : 10,
            "maxAPU" : 0,
            "maxNumOfTiles" : 8,
            "maxIriscPerTile" : 5,
            "IriscStartAddr" : 0xa001c,
            "IriscStep" : 0x200,
            "TileStart" : 0xc000000,
            "TileStep" : 0x800000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : True,
            # list of (trace type name, start_bit)
            "mask_classes": [("class1", 0), ("class2", 1)],
            "default_tracer_mode": "FIFO",
            "is_hca": False
        },
        {
            "name": "ConnectX6LX",
            "type": DeviceType.DEVICE_CONNECTX6LX,
            "dev_id": [0x216],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0x24780, 0, 16),
            "maxMainIrisc" : 8,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xe001c,
            "IriscStep" : 0x400,
            "TileStart" : 0x0,
            "TileStep" : 0x0,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": list(HCA_MASK_CLASSES),
            "default_tracer_mode": "MEM",
            "is_hca": True
        },
        {
            "name": "ConnectX7",
            "type": DeviceType.DEVICE_CONNECTX7,
            "dev_id": [0x218],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0x24780, 0, 16),
            "maxMainIrisc" : 11,
            "maxAPU" : 17,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xe001c,
            "IriscStep" : 0x400,
            "TileStart" : 0x0,
            "TileStep" : 0x0,
            "ApuStartAddr" : 0x99a000,#bit.16
            "ApuStep" : 0x800,
            "ApuMaxNumOfSteps" : 2, #based on amount of mask_trace_db in the APU
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": list(HCA_MASK_CLASSES),
            "default_tracer_mode": "MEM",
            "is_hca": True
        },
        {
            "name": "ConnectX8",
            "type": DeviceType.DEVICE_CONNECTX8,
            "dev_id": [0x21e],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0x24780, 0, 16),
            "maxMainIrisc" : 11,
            "maxAPU" : 17,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xe001c,
            "IriscStep" : 0x400,
            "TileStart" : 0x0,
            "TileStep" : 0x0,
            "ApuStartAddr" : 0x99a000,#bit.16
            "ApuStep" : 0x800,
            "ApuMaxNumOfSteps" : 2, #based on amount of mask_trace_db in the APU
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": list(HCA_MASK_CLASSES),
            "default_tracer_mode": "MEM",
            "is_hca": True
        },
        {
            "name": "Quantum2",
            "type": DeviceType.DEVICE_QUANTUM2,
            "dev_id": [0x257],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0xffffffff, 0, 16),
            "maxMainIrisc" : 10,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xa001c,
            "IriscStep" : 0x200,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": [("class1", 0), ("class2", 1)],
            "default_tracer_mode": "FIFO",
            "is_hca": False
        },
        {
            "name": "Quantum3",
            "type": DeviceType.DEVICE_QUANTUM3,
            "dev_id": [0xfaaf],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0xffffffff, 0, 16),
            "maxMainIrisc" : 10,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xa001c,
            "IriscStep" : 0x200,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": [("class1", 0), ("class2", 1)],
            "default_tracer_mode": "FIFO",
            "is_hca": False
        },
        {
            "name": "GB100",
            "type": DeviceType.DEVICE_GB100,
            "dev_id": [0xfeef],
            "chip_rev":-1,
            "embedded_fw_str_db": True,
            "fw_str_db_signature_exists": False,
            "fw_str_db_signature_addr": (0xffffffff, 0, 16),
            "maxMainIrisc" : 11,
            "maxAPU" : 0,
            "maxNumOfTiles" : 0,
            "maxIriscPerTile" : 0,
            "IriscStartAddr" : 0xffffffff,  #no info yet.
            "IriscStep" : 0xffffffff,
            "TileStart" : 0x2000000,
            "TileStep" : 0x200000,
            "ApuStartAddr" : 0x0,
            "ApuStep" : 0x0,
            "ApuMaxNumOfSteps" : 0,
            "supportPhyUc" : False,
            # list of (trace type name, start_bit)
            "mask_classes": [("class1", 0), ("class2", 1)],
            "default_tracer_mode": "FIFO",
            "is_hca": False
        },

    ]

    @classmethod
    def _cmd_exec(self, cmd):
        """
        Execute a command line and return the state and the output
        of the command
        """
        p = subprocess.Popen(cmd,
                             stdin=subprocess.PIPE,
                             stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE,
                             universal_newlines=False,
                             shell=True)
        output = p.communicate()
        stat = p.wait()
        return (stat, output[0].decode('utf-8'), output[1].decode('utf-8'))  # RC, Stdout, Stderr

    @classmethod
    def _is_dev_dbdf_format(self, dev):
        """
        Check if the input has a bdf format with domain
        If bdf format with domain format return True, otherwise return False
        """
        pat = r"[0-9,A-F,a-f]{4}:[0-9,A-F,a-f]{2}:[0-9,A-F,a-f]{2}\.[0-9,A-F,a-f]{1,2}"
        if re.match(pat, dev):
            return True
        return False

    @classmethod
    def _is_dev_bdf_format(self, dev):
        """
        Check if the input has a bdf format
        If bdf format return True, otherwise return False
        """
        pat = r"[0-9,A-F,a-f]{2}:[0-9,A-F,a-f]{2}\.[0-9,A-F,a-f]{1,2}"
        if re.match(pat, dev):
            return True
        return False

    @classmethod
    def _add_domain_to_address(self, dev_addr):
        """
        Add the domain "0000:" to the given address
        """
        if len(dev_addr.split(":")) == 2:
            return "0000:" + dev_addr
        return dev_addr

    @staticmethod
    def get_dev_dbdf(device_name):
        """
        retrieve the BDF according to the device name
        """
        if FwTraceUtilities._is_dev_dbdf_format(device_name):
            return device_name
        if FwTraceUtilities._is_dev_bdf_format(device_name):
            return FwTraceUtilities._add_domain_to_address(device_name)

        operatingSys = platform.system()

        if operatingSys == "FreeBSD":
            if not(device_name.startswith("pci")):
                raise RuntimeError("Unexpected device name format")
            return device_name[3:]
        elif operatingSys == "Linux":
            cmd = "mdevices_info -vv"
            (rc, out, _) = FwTraceUtilities._cmd_exec(cmd)
            if rc != 0:
                raise RuntimeError("Failed to get device PCI address")
            # extract bdf
            bdf = None
            for line in out.split('\n'):
                if device_name in line:
                    if len(line.split()) > 2:
                        bdf = line.split()[2]
            if not bdf:
                raise RuntimeError("Failed to get device PCI Address")
            return FwTraceUtilities._add_domain_to_address(bdf)
        elif operatingSys == "Windows":
            cmd = "mdevices status -vv"
            (rc, out, _) = FwTraceUtilities._cmd_exec(cmd)
            if rc != 0:
                raise RuntimeError("Failed to get device PCI address")
            # extract bdf
            bdf = None
            for line in out.split('\n'):
                l = line.split()
                if (len(l) > 1) and (device_name in l[0]) and ("seg:bus:dev.fn" in l[1]):
                    bdf = line.split('=')[1]
            if not bdf:
                raise RuntimeError("Failed to get device PCI Address")
            return bdf
        else:
            raise RuntimeError("Unsupported OS")

    @staticmethod
    def is_driver_mem_mode_supported():
        """
        Check if driver mem mode is suported
        If supported return True, otherwise return False
        """
        is_supported = False

        if os.name == "nt":
            # future capability - windows
            pass
        else:
            if os.path.exists("/sys/kernel/debug/tracing/events/mlx5/fw_tracer/") or \
               os.path.exists("/sys/kernel/debug/tracing/events/mlx5/mlx5_fw/"):
                is_supported = True

        return is_supported

    @staticmethod
    def is_secure_fw(mst_device):
        """
        Check if the firmware is secure or not
        If secure return True, otherwise return False
        """
        is_secure = False
        if mst_device is None:
            return is_secure

        # in case that reg access fail, we cant determain a secure fw and return false
        try:
            reg_access_obj = regaccess.RegAccess(mst_device)
            is_secure = reg_access_obj.getSecureFWStatus()
        except BaseException:
            is_secure = False

        return is_secure

    @staticmethod
    def ts_to_real_ts(ts, freq):
        """
        The method calculate a real time stamp
        and return is as [hh:mm:ss:nsec] format
        """
        if freq <= 0:
            raise RuntimeError("device frequency is not above Zero - can't calc real time stamp")

        nano_seconds = int(1000 / freq) * ts
        # calc hours
        time_in_seconds = int(nano_seconds / 1000000000)
        hours = int(time_in_seconds / 3600)
        # calc minutes
        time_in_seconds = time_in_seconds - (hours * 3600)
        minutes = int(time_in_seconds / 60)
        # calc seconds
        time_in_seconds = time_in_seconds - (minutes * 60)
        seconds = time_in_seconds
        # calc the reminder in nano seconds
        nsecs = nano_seconds - int(nano_seconds / 1000000000) * 1000000000

        return "{:02d}:{:02d}:{:02d}:{:09d}".format(hours, minutes, seconds, nsecs)

    @staticmethod
    def get_device_frequency(device):
        """
        The method call mlxuptime, parse
        the device frequency and return it as integer
        (mlxuptime can take 1 sec)
        """
        cmd = "mlxuptime -d" + device
        (rc, out, _) = FwTraceUtilities._cmd_exec(cmd)
        freq_num = 0
        if rc != 0:
            raise RuntimeError("Failed to get device frequency (necessary for real ts calculation)")
        else:
            for line in out.split('\n'):
                if "Measured core frequency" in line:
                    freq_str = line.split(": ")[1].split(" ")[0]
                    freq_num = int(round(float(freq_str)))
                    break

        return freq_num

##########################################


if __name__ == "__main__":
    sys.exit(ExtractStrings())
